(ns prism.internal.http-client-helidon
  (:require
    [clojure.string :as s]
    [prism.core :refer [defdelayed]])
  (:import
    (clojure.lang IPersistentCollection)
    (io.helidon.common.uri UriEncoding)
    (io.helidon.http Header HeaderNames Method)
    (io.helidon.webclient.api ClientRequest HttpClientRequest WebClient)
    (io.helidon.webclient.http1 Http1ClientResponse)
    (io.helidon.webclient.http2 Http2ClientResponse)
    (java.time Duration)
    (java.util Collection)
    (prism.internal HttpResponseDelegate)))

(defprotocol ^:private ->Array
  (->array ^"[Ljava.lang.String;" [_]))

(extend-protocol ->Array
  String
  (->array [this] (into-array String [this]))
  Collection
  (->array [^Collection this] (.toArray this))
  IPersistentCollection
  (->array [this] (into-array String this)))

(defprotocol ^:private HttpVersion
  (http-version [_]))

(extend-protocol HttpVersion
  Http1ClientResponse
  (http-version [_] :http-1.1)
  Http2ClientResponse
  (http-version [_] :http2))

(def http-methods
  {:get     Method/GET
   :post    Method/POST
   :put     Method/PUT
   :delete  Method/DELETE
   :head    Method/HEAD
   :options Method/OPTIONS
   :trace   Method/TRACE
   :patch   Method/PATCH
   :connect Method/CONNECT})

(defdelayed ^WebClient default-client (-> (WebClient/builder)
                                          .build))

(defn- add-headers! ^HttpClientRequest [^HttpClientRequest r m]
  (reduce-kv
    (fn add-header! [^ClientRequest r k v]
      (.header
        r
        (HeaderNames/create (name k))
        (->array (or v ""))))
    r
    m))

(defn- headers->map [headers]
  (persistent!
    (reduce
      (fn assoc-header! [acc ^Header header]
        (assoc! acc
                (-> (.name header) s/lower-case)
                (.get header)))
      (transient {})
      headers)))

(defn- execute-request! ^HttpResponseDelegate [^HttpClientRequest helidon-request {:keys [body] :as req}]
  (try
    (-> (if body
          (.submit helidon-request body)
          (.request helidon-request))
        HttpResponseDelegate.)
    (catch Exception e
      (throw (ex-info "Failed to execute request" req e)))))

(defn request [{:keys [url timeout request-method headers query-string http-client version] :as req}]
  (let [method (or (get http-methods request-method)
                   (some-> request-method name Method/create)
                   Method/GET)
        uri-string (cond-> (UriEncoding/encodeUri url)
                           (seq query-string) (str \? query-string))
        ^WebClient client (or http-client (default-client))
        resp (-> (.method client method)
                 (.skipUriEncoding true)
                 (.uri ^String uri-string)
                 (add-headers! headers)
                 (.protocolId (case version
                                :http2 "h2"
                                :http-1.1 "http/1.1"
                                "h2"))
                 (.readTimeout (Duration/ofMillis timeout))
                 (execute-request! req))]
    {:request      req
     :http-version (http-version (.delegate resp))
     :status       (.. resp status code)
     :body         (.inputStream resp)
     :headers      (-> (.headers resp)
                       headers->map)}))

(comment
  (request
    {:url            "https://www.google.com/"
     :request-method :get
     :version        :http-1.1
     :timeout        10000}))
