(ns prism.services
  (:require
    [prism.core :refer [defdelayed] :as prism]
    [prism.http :as http])
  (:import
    (java.io UncheckedIOException)
    (java.net ConnectException)))

(defn- host-port-url [{:keys [Address ServicePort]}]
  (format "http://%s:%s" Address ServicePort))

(defdelayed ^:private consul-svc (:consul-url (prism/config)))

(defdelayed ^:private static-services {:consul [(consul-svc)]})

(defn- lookup-svc [{:keys [service tag]}]
  (some->> (http/request (cond-> {:url (str (consul-svc) "/v1/catalog/service/" (name service))
                                  :as  :json}
                                 tag (assoc :query-params {:filter (format "ServiceTags contains \"%s\"" (name tag))})))
           :body
           (mapv host-port-url)))

(defdelayed ^:private service-registry (:services (prism/config)))

(defdelayed ^:private services (atom (merge (static-services)
                                            (:service-defaults (prism/config)))))

(defn- resolve-service! [service-kw]
  (when-let [resolved (-> (service-kw (service-registry))
                          lookup-svc
                          not-empty)]
    (-> (swap! (services) assoc service-kw resolved)
        service-kw)))

(defn- service! [service-kw]
  (or (service-kw @(services))
      (resolve-service! service-kw)))

(defn invalidate-service! [service-kw]
  (when-not (service-kw (static-services))
    (swap! (services) dissoc service-kw)))

(defn service-url! [svc]
  (-> (service! svc)
      rand-nth))

(defn- pathed-service-url! [service path]
  (some-> (service-url! service)
          (str path)))

(defn- handle-connect-exception! [service path req previous throw?]
  (invalidate-service! service)
  (if-let [url (pathed-service-url! service path)]
    (http/request (assoc req :url url))
    (cond-> (ex-info "Could not resolve domain name for service" {:service  service
                                                                  :status   503
                                                                  :previous previous})
            throw? throw)))

(defn service-request [service path {:keys [catch-exceptions?] :as req}]
  (if-let [url (pathed-service-url! service path)]
    (if catch-exceptions?
      (let [result (http/request (assoc req :url url))]
        (if (prism/ex-caused-by? result ConnectException)
          (handle-connect-exception! service path req url false)
          result))
      (try
        (http/request (assoc req :url url))
        (catch Exception ex
          (if (prism/ex-caused-by? ex ConnectException)
            (handle-connect-exception! service path req url true)
            (throw ex)))))
    (cond-> (ex-info "Could not resolve domain name for service"
                     {:service service
                      :path    path
                      :status  503})
            (not catch-exceptions?) throw)))

(comment
  (-> (prism/config)
      :services
      vals
      first
      lookup-svc))
