(ns prism.core
  (:require
    [clojure.java.io :as io]
    [clojure.string :as s]
    [prism.internal.classpath :as cp])
  (:import
    (clojure.lang Delay)
    (java.util.concurrent CompletableFuture)))

(defmacro defdelayed [name body]
  (let [delayed-name-sym (gensym (str name "-delayed"))]
    `(do
       (def ~(with-meta delayed-name-sym {:tag      Delay
                                          :private  true
                                          :arglists [[]]}) (delay ~body))
       (defn ~name [] (.deref ~delayed-name-sym)))))

(defdelayed prism-version (-> (io/resource "prism/version.txt")
                              slurp))

(defn deep-merge [& maps]
  (letfn [(reconcile-keys [val-in-result val-in-latter]
            (if (and (map? val-in-result)
                     (map? val-in-latter))
              (merge-with reconcile-keys val-in-result val-in-latter)
              val-in-latter))
          (reconcile-maps [result latter]
            (merge-with reconcile-keys result latter))]
    (reduce reconcile-maps maps)))

(declare config)
(cp/when-ns 'aero.core
  (defmethod aero.core/reader 'deep-merge
    [_ _ value]
    (apply deep-merge value))

  (defdelayed config (-> (io/resource "config.edn")
                         aero.core/read-config)))

(defmacro try-or
  ([] nil)
  ([x] x)
  ([x & next]
   `(try ~x (catch Exception _# (try-or ~@next)))))

(defmacro kw-map [& vars]
  (into {} (map (juxt keyword identity)) vars))

(declare named-format)
(cp/when-ns 'com.gfredericks.like-format-but-with-named-args
  (defmacro named-format [s arg & args]
    (assert (if (symbol? arg)
              (every? symbol? args)
              (and (or (map? arg) (seq? arg))
                   (empty? args)))
            "expects either 1 map or 1+ symbols")
    (let [args-map (if (symbol? arg)
                     (macroexpand `(kw-map ~@(conj args arg)))
                     arg)]
      (list com.gfredericks.like-format-but-with-named-args/named-format s args-map))))

(defn load-version []
  (-> (slurp "version.txt")
      s/trimr))

(defn assoc-by [f coll]
  (-> (reduce #(assoc! %1 (f %2) %2)
              (transient {})
              coll)
      persistent!))

(defn assoc-by-kw [kw coll]
  (-> (reduce #(assoc! %1 (kw %2) (dissoc %2 kw))
              (transient {})
              coll)
      persistent!))

(defn- update-text [?kw f]
  (when ?kw
    (cond->> (f (name ?kw))
             (keyword? ?kw) (keyword (namespace ?kw)))))

(defn snake->kebab [k]
  (update-text k #(s/escape % {\_ \-})))

(defn kebab->snake [k]
  (update-text k #(s/escape % {\- \_})))

(defn vec-first [coll]
  (nth coll 0))

(defn ex-caused-by? [ex clz]
  (or (instance? clz ex)
      (some-> (ex-cause ex)
              (recur clz))))

(defn complete-all! [coll]
  (-> (into-array CompletableFuture coll)
      CompletableFuture/allOf
      .join)
  (mapv ^[] CompletableFuture/.get coll))

(comment
  (config)

  (named-format "Hello my name is %name~s and I am %height~.3f feet tall."
                {:name   "Joe Biden"
                 :height (* 2 Math/PI)})
  (named-format "Hello my name is %name~s and I am %height~.3f feet tall."
                (assoc {:name   "Joe Biden"
                        :height (* 2 Math/PI)}
                  :name "test"))
  (->> (assoc {:name   "Joe Biden"
               :height (* 2 Math/PI)}
         :name "test")
       (named-format "Hello my name is %name~s and I am %height~.3f feet tall."))
  (let [x {:name   "Joe Biden"
           :height (* 2 Math/PI)}]
    (named-format "Hello my name is %name~s and I am %height~.3f feet tall."
                  (do x)))
  (let [name "Joe Biden 2"
        height (* 2 Math/PI)]
    (named-format "Hello my name is %name~s and I am %height~.3f feet tall."
                  name height)))
