(ns prism.http
  (:require
    [clojure.java.io :as io]
    [hato.client :as hato]
    [hato.conversion :as hc]
    [hato.middleware :as hmw]
    [prism.core :as prism]
    [prism.internal.aws-sign :as aws-sign]
    [prism.internal.classpath :as cp])
  (:import
    (java.io InputStream Reader)
    (java.net ConnectException)
    (java.net.http HttpRequest$BodyPublishers)))

(def ^:private implementation
  (or (cp/when-ns 'prism.apache-http-client
        {:fn   prism.apache-http-client/request*
         :type :apache})
      {:fn   hato/request*
       :type :hato}))

(def ^:private wrap-retry-request nil)
(declare ^:private attempt-request)
(cp/when-ns 'again.core
  (cp/when-ns 'taoensso.timbre
    (let [cb (fn [{:again.core/keys [exception status]}]
               (when (#{:retry :attempt} status)
                 (let [data (ex-data exception)
                       request (-> data :request (select-keys [:url :method]))
                       response (-> data (select-keys [:status :body]))]
                   (taoensso.timbre/with-context
                     (merge request response)
                     (taoensso.timbre/warnf exception "Error while making HTTP request."))))
               (when (or (-> exception
                             ex-data
                             :status
                             (= 400))
                         (instance? ConnectException exception))
                 :again.core/fail))
          retry-config (fn [{request-retry :retry}]
                         {:again.core/callback cb
                          :again.core/strategy (cond
                                                 (vector? request-retry) request-retry
                                                 (number? request-retry) (take request-retry (list* 1000 2000 (repeat 3000)))
                                                 (true? request-retry) [1000 2000 3000 3000])})]

      (defn- wrap-retry-request [client]
        (fn
          ([req]
           (if (:retry req)
             (again.core/with-retries
               (retry-config req)
               (client req))
             (client req)))
          ([req respond raise]
           (if (:retry req)
             (client req
                     #(again.core/with-retries
                        (retry-config req)
                        (respond %))
                     raise)
             (client req respond raise))))))))

(defn- attempt-request [f req]
  (try
    (f req)
    (catch Exception e
      (cp/when-ns 'taoensso.timbre
        (taoensso.timbre/debugf "HTTP Exception: %s" e))
      e)))

(defn- wrap-catch-exceptions [client]
  (fn
    ([req]
     (if (:catch-exceptions? req)
       (attempt-request client req)
       (client req)))
    ([req respond raise]
     (if (:catch-exceptions? req)
       (client req #(attempt-request respond %) raise)
       (client req respond raise)))))

(defn- default-user-agent [req]
  (update-in req [:headers "user-agent"]
             (fnil identity (str "Prism " (prism/prism-version)))))

(defn- wrap-user-agent [client]
  (fn
    ([req]
     (-> (default-user-agent req)
         (client)))
    ([req respond raise]
     (client (default-user-agent req) respond raise))))

(defn- resp->charset [resp] (or (-> resp :content-type-params :charset) "UTF-8"))

(defn- coerce-response-body [{:keys [coerce]} {:keys [body status] :as resp} coerce-fn]
  (let [^String charset (resp->charset resp)]
    (cond
      (and (hmw/unexceptional-status? status)
           (or (nil? coerce) (= coerce :unexceptional)))
      (with-open [r (io/reader body :encoding charset)]
        (assoc resp :body (coerce-fn r)))
      (= coerce :always)
      (with-open [r (io/reader body :encoding charset)]
        (assoc resp :body (coerce-fn r)))
      (and (not (hmw/unexceptional-status? status)) (= coerce :exceptional))
      (with-open [r (io/reader body :encoding charset)]
        (assoc resp :body (coerce-fn r)))
      :else resp)))

(def ^:private wrap-json-body nil)
(cp/when-ns 'prism.json
  (defn- coerce-json-response-body [req resp]
    (coerce-response-body req resp prism.json/json->clj))
  (defmethod hmw/coerce-response-body :json [req resp] (coerce-json-response-body req resp))
  (defmethod hmw/coerce-response-body :json-strict [req resp] (coerce-json-response-body req resp))

  (defmethod hc/decode :application/json
    [resp _]
    (let [^String charset (resp->charset resp)]
      (-> (:body resp)
          (io/reader :encoding charset)
          prism.json/json->clj)))

  (defn- wrap-json-body [client]
    (let [add-json-body (fn [req]
                          (if-some [body-json (:json req)]
                            (-> (assoc req :body (prism.json/write-json-string body-json)
                                           :content-type :json)
                                (dissoc :json))
                            req))]
      (fn
        ([req]
         (-> (add-json-body req)
             client))
        ([req respond raise]
         (-> (add-json-body req)
             (client respond raise)))))))

(cp/when-class 'org.json.XML
  (declare ^:private xml->map)
  (defn- xml->val [xml-val]
    (cond
      (= org.json.JSONObject/NULL xml-val) nil
      (instance? org.json.JSONObject xml-val) (xml->map xml-val)
      (instance? org.json.JSONArray xml-val) (mapv xml->val xml-val)
      :else xml-val))

  (defn- xml->map [^org.json.JSONObject jo]
    (-> (reduce
          (fn [m k]
            (->> (.get jo k)
                 xml->val
                 (assoc!
                   m
                   (keyword k))))
          (transient {})
          (.keySet jo))
        persistent!))

  (defn- xml->clj [r]
    (let [jo (org.json.XML/toJSONObject ^Reader r)]
      (xml->map jo)))

  (defmethod hmw/coerce-response-body :xml [req resp]
    (coerce-response-body req resp xml->clj))

  (defmethod hc/decode :application/xml
    [resp _]
    (let [^String charset (resp->charset resp)]
      (-> (:body resp)
          (io/reader :encoding charset)
          xml->map))))

(defn- sign-aws-request [request]
  (if-let [aws-config (:aws request)]
    (-> (aws-sign/sign-request request aws-config)
        (dissoc :aws))
    (dissoc request :aws)))

(defn- wrap-aws-sign
  ([client]
   (fn
     ([request]
      (-> (sign-aws-request request)
          client))
     ([request respond raise]
      (-> (sign-aws-request request)
          (client respond raise))))))

(defn- remove-hop-to-hop-headers [response]
  (update
    response
    :headers
    dissoc
    ;https://datatracker.ietf.org/doc/html/rfc2616#section-13.5.1
    "connection"
    "keep-alive"
    "proxy-authenticate"
    "proxy-authorization"
    "te"
    "trailers"
    "transfer-encoding"
    "upgrade"))

(defn- proxy-request [request]
  (-> (update request :as (fnil identity :stream))
      (update :decompress-body (fnil identity false))
      (update :throw-exceptions (fnil identity false))
      (update :remove-hop-to-hop-headers? (fnil identity true))))

(defn- wrap-proxy
  ([client]
   (fn
     ([request]
      (-> (dissoc request :proxy?)
          (cond-> (:proxy? request) proxy-request)
          client))
     ([request respond raise]
      (-> (dissoc request :proxy?)
          (cond-> (:proxy? request) proxy-request)
          (client respond raise))))))

(defn- wrap-remove-hop-to-hop-headers
  ([client]
   (fn
     ([request]
      (-> (client request)
          (cond-> (:remove-hop-to-hop-headers? request) remove-hop-to-hop-headers)))
     ([request respond raise]
      (client request
              (cond-> respond
                      (:remove-hop-to-hop-headers? request) (comp remove-hop-to-hop-headers))
              raise)))))

(defn- wrap-fix-headers
  "https://github.com/gnarroway/hato/issues/59"
  [client]
  (fn
    ([req]
     (-> (client req)
         (update :headers dissoc ":status")))
    ([req respond raise]
     (client req #(respond (update % :headers dissoc ":status")) raise))))

(defn- stream->publisher [stream content-length]
  (HttpRequest$BodyPublishers/fromPublisher
    (HttpRequest$BodyPublishers/ofInputStream (fn [] stream))
    content-length))

(defn- stream-request->publisher-request [{:keys [body content-length] :as req}]
  (cond-> req
          (and content-length (instance? InputStream body)) (assoc :body (stream->publisher body content-length))))

(defn- wrap-stream-body [client]
  (fn
    ([req]
     (-> (stream-request->publisher-request req)
         client))
    ([req respond raise]
     (-> (stream-request->publisher-request req)
         (client respond raise)))))

(def ^:private wrapped-request (->> [(when (= :hato (:type implementation))
                                       wrap-stream-body)
                                     wrap-json-body
                                     wrap-retry-request
                                     wrap-user-agent
                                     wrap-remove-hop-to-hop-headers
                                     (when (= :hato (:type implementation))
                                       wrap-fix-headers)
                                     wrap-proxy
                                     wrap-catch-exceptions]
                                    (into hmw/default-middleware (filter some?))
                                    (into [wrap-aws-sign])
                                    (hmw/wrap-request (:fn implementation))))

(defn request [req]
  (let [req (update req :timeout (fnil identity 10000))]
    (if (:async? req)
      (wrapped-request req (:respond req identity) (:raise req #(throw %)))
      (wrapped-request req))))

(comment
  (defn- gen-string [size]
    (apply str (repeatedly size #(rand-nth "abcdefghijklmnopqrstuvwxyz0123456789"))))
  (do (request {:url     "https://webhook.site/1db28338-bb65-450c-9132-04f0feb29cb8"
                :method  :post
                :timeout Long/MAX_VALUE
                :body    (java.io.StringBufferInputStream. (gen-string 19999 #_20000))
                :headers {}})
      nil)
  (do (request {:url     "http://localhost:3003/test"
                :method  :post
                :timeout Long/MAX_VALUE
                :body    (java.io.StringBufferInputStream. (gen-string 16384 #_20000))
                :headers {}})
      nil)
  (request {:url          "https://github.com"
            :method       :get
            :proxy?       true
            :query-params {:query "xyz"}}))
