(ns prism.internal.aws-sign
  "Modified from https://github.com/jacobemcken/aws-simple-sign"
  (:require
    [clojure.string :as str])
  (:import
    (java.net URI)
    (java.security MessageDigest)
    (java.time ZoneId ZoneOffset)
    (java.time.format DateTimeFormatter)
    (java.util Date)
    (javax.crypto Mac)
    (javax.crypto.spec SecretKeySpec)))

(defn ^:no-doc hash-sha256
  [^bytes input]
  (let [hash (MessageDigest/getInstance "SHA-256")]
    (.update hash input)
    (.digest hash)))

(def ^:no-doc digits
  (char-array "0123456789abcdef"))

(defn ^:no-doc hex-encode
  [bytes]
  (->> bytes
       (mapcat #(list (get digits (bit-shift-right (bit-and 0xF0 %) 4))
                      (get digits (bit-and 0x0F %))))))

(defn ^:no-doc hex-encode-str
  [bytes]
  (->> (hex-encode bytes)
       (apply str)))

;; Clojure implementation of signature
;; https://gist.github.com/souenzzo/21f3e81b899ba3f04d5f8858b4ecc2e9
;; https://github.com/joseferben/clj-aws-sign/ (ring middelware)

(defn ^:no-doc hmac-sha-256
  [key ^String data]
  (let [algo "HmacSHA256"
        mac (Mac/getInstance algo)]
    (.init mac (SecretKeySpec. key algo))
    (.doFinal mac (.getBytes data "UTF-8"))))

(defn ^:no-doc char-range
  [start end]
  (mapv char (range (int start) (inc (int end)))))

(def ^:no-doc unreserved-chars
  (->> (concat '(\- \. \_ \~)
               (char-range \A \Z)
               (char-range \a \z)
               (char-range \0 \9))
       (into #{})))

(def ^:no-doc url-unreserved-chars
  (conj unreserved-chars \/))

(defn ^:no-doc encode
  [skip-chars c]
  (if (skip-chars c)
    c
    (let [byte-val (int c)]
      (format "%%%X" byte-val))))

(defn ^:no-doc uri-encode
  [skip-chars uri]
  (->> (mapv (partial encode skip-chars) uri)
       (apply str)))

(def ^DateTimeFormatter ^:no-doc formatter
  (-> (DateTimeFormatter/ofPattern "yyyyMMdd'T'HHmmss'Z'")
      (.withZone (ZoneId/from ZoneOffset/UTC))))

(defn ^:no-doc compute-signature
  [{:keys [credentials str-to-sign region service short-date]}]
  (-> (str "AWS4" (:aws/secret-access-key credentials))
      (.getBytes)
      (hmac-sha-256 short-date)
      (hmac-sha-256 region)
      (hmac-sha-256 service)
      (hmac-sha-256 "aws4_request")
      (hmac-sha-256 str-to-sign)
      hex-encode-str))

(def ^:no-doc algorithm "AWS4-HMAC-SHA256")

(defn signature
  "AWS specification: https://docs.aws.amazon.com/AmazonS3/latest/API/sig-v4-header-based-auth.html

   Inspired by https://gist.github.com/souenzzo/21f3e81b899ba3f04d5f8858b4ecc2e9"
  [credentials canonical-url {:keys [scope method timestamp region service query-string content-sha256 signed-headers]}]
  (let [encoded-url (uri-encode url-unreserved-chars canonical-url)
        signed-headers (into (sorted-map) (map (fn [[k v]] [(str/lower-case k) v])) signed-headers)
        headers-str (->> (mapv (fn [[k v]] (str k ":" (str/trim (or v "")) "\n")) signed-headers)
                         (apply str))
        signed-headers-str (str/join ";" (mapv key signed-headers))
        canonical-request (str (or method "GET") "\n"
                               encoded-url "\n"
                               query-string "\n"
                               headers-str "\n"
                               signed-headers-str "\n"
                               (or content-sha256 "UNSIGNED-PAYLOAD"))
        str-to-sign (str algorithm "\n"
                         timestamp "\n"
                         scope "\n"
                         (hex-encode-str (hash-sha256 (.getBytes canonical-request))))]
    (compute-signature {:credentials credentials
                        :str-to-sign str-to-sign
                        :region      region
                        :service     service
                        :short-date  (subs timestamp 0 8)})))

(defn ^:no-doc hashed-payload
  [payload]
  (cond
    (nil? payload) (hex-encode-str (hash-sha256 (.getBytes "")))
    (string? payload) (hex-encode-str (hash-sha256 (.getBytes ^String payload)))
    (bytes? payload) (hex-encode-str (hash-sha256 payload))
    :else "UNSIGNED-PAYLOAD"))

(defn- canonise-query-string [query-string]
  (cond-> query-string ;add = for /?<path-section> requests, e.g. https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjects.html
          (and (seq query-string)
               (not (str/includes? query-string "="))) (str "=")))

(defn sign-request
  "Takes a client and a Ring style request map.
   Returns an enriched Ring style map with the required headers
   needed for AWS signing."
  ([{:keys [body headers request-method url query-string] :as request} ;assumes query-string is built from sorted map
    {:keys [credentials ref-time region service]
     :or   {service  "execute-api"
            ref-time (Date.)}}]
   (let [timestamp (.format formatter (.toInstant ^Date ref-time))
         url-obj (-> (URI. url)
                     .toURL)
         port (.getPort url-obj)
         content-sha256 (hashed-payload body)
         session-token (:aws/session-token credentials)
         signed-headers (cond-> (assoc headers
                                  "Host" (cond-> (.getHost url-obj)
                                                 (pos? port) (str \: port))
                                  "x-amz-content-sha256" content-sha256
                                  "x-amz-date" timestamp)
                                session-token (assoc "x-amz-security-token" session-token))
         scope (str (subs timestamp 0 8) "/" region "/" service "/aws4_request")
         signature-str (signature credentials
                                  (.getPath url-obj)
                                  {:scope          scope
                                   :timestamp      timestamp
                                   :region         region
                                   :service        service
                                   :method         (-> request-method name str/upper-case)
                                   :signed-headers signed-headers
                                   :query-string   (canonise-query-string query-string)
                                   :content-sha256 content-sha256})
         headers (-> (dissoc signed-headers "Host")
                     (assoc "Authorization" (str algorithm " Credential=" (:aws/access-key-id credentials) "/" scope ", "
                                                 "SignedHeaders=" (str/join ";" (map key signed-headers)) ", "
                                                 "Signature=" signature-str))
                     (update-keys str/lower-case))]
     (assoc request :headers headers))))
