(ns prism.logging
  (:require
    [clojure.stacktrace :as stacktrace]
    [clojure.string :as str]
    [prism.core :as prism]
    [prism.internal.classpath :as cp]
    [prism.json :as json]
    [taoensso.encore]
    [taoensso.telemere :as tel]
    [taoensso.trove :as trove]
    [taoensso.trove.telemere :as trove-telemere]))

(defn structure-timbre-log [{:keys [msg_ ?err ?ns-str ?line level timestamp_ context data]}]
  (let [thread (Thread/currentThread)
        thread-str (or (not-empty (.getName thread))
                       (str "virtual-thread-" (.threadId thread)))]
    (cond-> (assoc context
              :caller (str ?ns-str \: ?line)
              :level (name level)
              :thread thread-str
              :msg (force msg_)
              :ts (force timestamp_))
            data (assoc :data data)
            ?err (assoc :stacktrace (with-out-str
                                      (stacktrace/print-cause-trace ?err)))
            (seq (ex-data ?err)) (assoc :ex-data (ex-data ?err)))))

(defn structure-telemere-log [{:keys [inst ns coords level id uid msg_ data error ctx thread]}]
  (let [ex-data (ex-data error)]
    (cond-> {:caller (str ns \: (str/join \: coords))
             :level  (name level)
             :ts     inst
             :thread (or (:name thread)
                         (str (:group thread "virtual-thread")
                              "-"
                              (:id thread)))
             :msg    (force msg_)}
            data (assoc :data data)
            id (assoc :id id)
            uid (assoc :uid uid)
            ctx (assoc :context ctx)
            error (assoc :stacktrace (with-out-str
                                       (stacktrace/print-cause-trace error)))
            ex-data (assoc :ex-data ex-data))))

(defn configure-telemere-logging! []
  (-> (prism/config) :log-level keyword tel/set-min-level!)
  (tel/remove-handler! :default/console)
  (cp/when-ns 'prism.json
    (tel/add-handler!
      ::json-console-handler
      (tel/handler:console
        {:stream    :err
         :output-fn (tel/pr-signal-fn {:clean-fn identity
                                       :pr-fn    (fn pr-log [signal]
                                                   (some-> (structure-telemere-log signal)
                                                           json/write-json-string))})})
      {:track-stats? false}))
  (trove/set-log-fn! (trove-telemere/get-log-fn)))

(def configure-timbre-logging! (cp/missing-dep "jsonista" "aero" "timbre"))
(cp/when-ns 'taoensso.timbre
  (cp/when-ns 'prism.json
    (cp/when-ns 'aero.core
      (defn configure-timbre-logging! []
        (do (configure-telemere-logging!)
            (taoensso.timbre/set-config!
              {:min-level      (-> (prism/config) :log-level keyword)
               :ns-filter      #{"*"}
               :middleware     [(fn structure-log-middleware [m]
                                  (println 'structure-log-middleware)
                                  {:output_ (-> (structure-timbre-log m)
                                                prism.json/write-json-string
                                                delay)})]
               :timestamp-opts taoensso.timbre/default-timestamp-opts
               :appenders      {:println (taoensso.telemere.timbre/timbre->telemere-appender)}}))))))

(comment
  (configure-telemere-logging!)
  (configure-timbre-logging!)
  (tel/log! {:level :warn
             :data  {:m 1}
             :id    ::x}
            "test")
  (tel/log! {:level :debug
             :id    ::y}
            "test")
  (tel/log!
    {:level :info
     :error (Exception.)
     :id    ::y}
    "test"))
