(ns prism.logging
  (:require
    [clojure.stacktrace :as stacktrace]
    [clojure.string :as str]
    [prism.core :as prism]
    [prism.json :as json]
    [taoensso.encore]
    [taoensso.telemere :as tel]
    [taoensso.trove :as trove]
    [taoensso.trove.telemere :as trove-telemere]))

(defn structure-telemere-log [{:keys [inst ns coords level id uid msg_ data error ctx thread]}]
  (let [ex-data (ex-data error)]
    (cond-> {:caller (str ns \: (str/join \: coords))
             :level  (name level)
             :ts     inst
             :thread (or (:name thread)
                         (str (:group thread "virtual-thread")
                              "-"
                              (:id thread)))
             :msg    (force msg_)}
            data (assoc :data data)
            id (assoc :id id)
            uid (assoc :uid uid)
            ctx (assoc :context ctx)
            error (assoc :stacktrace (with-out-str
                                       (stacktrace/print-cause-trace error)))
            ex-data (assoc :ex-data ex-data))))

(defn configure-telemere-logging! []
  (-> (prism/config) :log-level keyword tel/set-min-level!)
  (tel/remove-handler! :default/console)
  (tel/add-handler!
    ::json-console-handler
    (tel/handler:console
      {:stream    :err
       :output-fn (tel/pr-signal-fn {:clean-fn identity
                                     :pr-fn    (fn pr-log [signal]
                                                 (some-> (structure-telemere-log signal)
                                                         json/write-json-string))})})
    {:track-stats? false})
  (trove/set-log-fn! (trove-telemere/get-log-fn)))

(comment
  (configure-telemere-logging!)
  (configure-timbre-logging!)
  (tel/log! {:level :warn
             :data  {:m 1}
             :id    ::x}
            "test")
  (tel/log! {:level :debug
             :id    ::y}
            "test")
  (tel/log!
    {:level :info
     :error (Exception.)
     :id    ::y}
    "test"))
