(ns atombind.core
  (:require [goog.dom :as d]
            [goog.array :as a]
            [goog.dom.classlist :as cl]
            [goog.events :as e]
            [goog.style :as st]
            [goog.events.EventType :as EventType]))

(enable-console-print!)

;;
;; Constants
;; 

(def ^:private BINJ-LOOP-PERIOD 30)
(def bind-scope "bind-scope")
(def bind-each "bind-each")
(def bind-show "bind-show")
(def bind-attr "bind-attr")
(def bind-func "bind-func")
(def bind "bind")
(def bound-to-coll "bound-to-coll")

(defn- err
  [msg]
  (println msg))

;;
;; Protocols and extensions
;;

(defprotocol IBindable
  (-bind! [el]))

(defprotocol IRenderable
  (-render! [el data-atom])
  (-render-all! [el data-atom] "Renders all sub-elements")
  (-set-attrs! [el data-atom] "Sets attributes according to data-binding")
  (-render-each! [el data-atom] "Repeats the element for each item in the bound collection"))

(defprotocol IHideable
  (-hide-or-show! [el data-atom] "Hides an element if it's 'bind-show' evaluates to false."))

(defprotocol IStringExt
  (-starts-with? [str prefix]))

(extend-type js/String
  IStringExt
  (-starts-with? [str prefix]
    "Taken from http://stackoverflow.com/questions/646628/how-to-check-if-a-string-startswith-another-string"
    (identical? 0 (.lastIndexOf str prefix 0))))

(extend-type js/NodeList
  ISeqable
  (-seq [nl] (array-seq nl)))

(defn- str-to-mappings
  [str]
  (let [split-regex #"([\w-_]+\|[\w-_]+|[\w-_]+\|\{\{.*?\}\})\s+"
        blank? (fn [s] (= (.trim s) ""))]
    (->> (js->clj (.split str split-regex))
         (filter (complement blank?))
         (map #(js->clj (.split % "|"))))))

(defn- evaluate
  [data binding]
  (let [expr-regex #"^\{\{(.*?)\}\}$"
        match-result (.match binding expr-regex)]
    (if match-result
      (let [inner (aget match-result 1)
            inner-binding-regex (js/RegExp. "\\%([\\w_-]+?)\\%" "g")
            inner-bindings (.match inner inner-binding-regex)
            new-inner (reduce (fn [s inner-binding]
                                (let [expr (.replace inner-binding
                                                     (js/RegExp. "\\%" "g")
                                                     "")
                                      evald (evaluate data expr)
                                      string? (= (type evald) js/String)]
                                  (.replace s
                                            (js/RegExp. inner-binding "g")
                                            (if string?
                                              (str "\"" evald "\"")
                                              evald))))
                              inner
                              inner-bindings)
            js-str (str "(function(){return " new-inner ";})();")]
        (try
          (js/eval js-str)
          (catch js/Error e
            (err (str "Error evaluating statement: "
                      new-inner
                      "\nError: "
                      e))
            nil)))
      (get data (keyword binding)))))

(defn- add-change-handler!
  [el data-atom]
  (when-not (cl/contains el "binj-change-handler-registered")
    (cl/add el "binj-change-handler-registered")
    (e/listen el
              (array EventType/CHANGE EventType/KEYUP)
              (fn [_]
                (when-not (cl/contains el "binj-dirty")
                  (cl/add el "binj-dirty")
                  (let [bind-key (keyword (.getAttribute el "bind"))]
                    (swap! data-atom assoc bind-key (.-value el)))
                  (js/setTimeout (fn [] (cl/remove el "binj-dirty")) BINJ-LOOP-PERIOD))))))

(defn- bind-event-handlers!
  [el data-atom]
  (when-not (cl/contains el "binj-bind-handlers-registered")
    (cl/add el "binj-bind-handlers-registered")
    (let [binding-string (.getAttribute el bind-func)
          bindings (str-to-mappings binding-string)]
      (doseq [[event func-name] bindings]
        (e/listen el event (evaluate @data-atom func-name))))))

(extend-type js/HTMLInputElement
  IRenderable
  (-render! [el data-atom]
    (when-not (cl/contains el "binj-dirty")
      (when-let [content (->> (.getAttribute el "bind")
                              keyword
                              (get @data-atom))]
        (set! (.-value el) content)
        (add-change-handler! el data-atom)))))

(defn- random
  []
  (.floor js/Math (.random js/Math 100)))

(defn- clean-element!
  "Removes all pseudo-classes and data that have been added to an element (and its children)."
  [el]
  (cl/remove el "binj-bind-handlers-registered")
  (doseq [e (d/getChildren el)]
    (clean-element! e)))

;; TODO Make collection re-rendering more efficient. Right now, every time the atom changes, the collections are re-rendered.
(defn- unrender-collections!
  [el]
  (let [els (.querySelectorAll el (str "[" bound-to-coll "]"))
        templates (.querySelectorAll el (str "[" bind-each "]"))]
    (doseq [x els]
      (d/removeNode x))
    (doseq [x templates]
      (clean-element! x))))

(extend-protocol ISeqable

  js/HTMLCollection
  (-seq [coll]
    (cond (= 0 (.-length coll)) nil
          :else (lazy-seq
                 (cons (aget coll 0) (-seq (a/slice coll 1))))))

  js/Array
  (-seq [coll]
    (cond (= 0 (.-length coll)) nil
          :else (lazy-seq
                 (cons (aget coll 0) (-seq (a/slice coll 1)))))))

(defn- is-scope?
  [el]
  (let [has-attr? (fn
                    [s]
                    (not (nil? (.getAttribute el s))))]
    (or (has-attr? bind-scope)
        (has-attr? bind-each))))

(defn- same-scope?
  [outer-elem inner-elem]
  "Returns whether or not inner-elem belongs in the same scope as outer elem."
  (loop [e inner-elem]
    (cond (nil? e) true
          (identical? e outer-elem) true
          (is-scope? e) false
          :else (recur (d/getParentElement e)))))

(extend-type js/HTMLElement
  IBindable
  (-bind! [el]
    (let [scope-name (.getAttribute el "bind-scope")
          bind-target (try
                        (js/eval scope-name)
                        (catch js/Error e
                          (err (str "Could not resolve scope: " scope-name "\nError: " e))
                          nil))]
      (when bind-target
        (-render-all! el bind-target))
      {:scope scope-name
       :element el
       :atom-ref (or bind-target (atom nil))}))
  IRenderable
  (-render! [el data-atom]
    (when-let [content (->> (.getAttribute el "bind")
                            keyword
                            (get @data-atom))]
      (d/setTextContent el content)))
  (-render-all! [el data-atom]
    (let [find-and-do (fn [attr-name f & [include-top-level? _]]
                        (doseq [e (.querySelectorAll el (str "[" attr-name "]"))]
                          (when (same-scope? el e)
                            (let [val (.getAttribute e attr-name)]
                              (when val
                                (f e data-atom)))))
                        (when include-top-level?
                          (let [val (.getAttribute el attr-name)]
                            (when val
                              (f el data-atom)))))
          render-collections! (fn []
                                (doseq [e (.querySelectorAll el (str "[" bind-each "]"))]
                                  (-render-each! e data-atom)))]
      (unrender-collections! el)
      (find-and-do "bind" -render!)
      (find-and-do "bind-show" -hide-or-show!)
      (find-and-do "bind-func" bind-event-handlers! true)
      (find-and-do bind-attr -set-attrs! true)
      (render-collections!)))
  (-set-attrs! [el data-atom]
    (let [attr-bindings (str-to-mappings (.getAttribute el bind-attr))]
      (doseq [[attr binding] attr-bindings]
        (let [value (evaluate @data-atom binding)]
          (when-let [to-set (if (= js/Function (type value))
                              (value el)
                              value)]
            (.setAttribute el attr to-set))))))
  (-render-each! [el data-atom]
    (let [coll-ref (.getAttribute el bind-each)
          parent (d/getParentElement el)
          coll (get @data-atom (keyword coll-ref))
          template (let [x (.cloneNode el true)]
                     (.removeAttribute x bind-each)
                     (.setAttribute x bound-to-coll coll-ref)
                     x)]
      (when (seq coll)
        (st/setElementShown el true)
        (loop [last-node nil
               first? (nil? last-node)
               new-node (.cloneNode el true)
               data (first coll)
               remaining-data (rest coll)]
          (-render-all! new-node (atom data)) ;; TODO Should not have to wrap data in an atom here. Render fns should take care of it.
          (if first?
            (d/replaceNode new-node el)
            (d/insertSiblingAfter new-node last-node))
          (when (seq remaining-data)
            (recur new-node
                   false
                   (.cloneNode template true)
                   (first remaining-data)
                   (rest remaining-data)))))
      (when-not (seq coll)
        (st/setElementShown el false))))
  IHideable
  (-hide-or-show! [el data-atom]
    (let [show-expr (.getAttribute el "bind-show")
          is-negated? (-starts-with? show-expr "!")
          show-keyword (keyword (if is-negated?
                                  (.substr show-expr 1)
                                  show-expr))
          show-val (show-keyword @data-atom)
          should-show? (if is-negated?
                         (not show-val)
                         show-val)]
      (st/setElementShown el should-show?))))

(defn- update-scopes
  [m {:keys [scope element atom-ref]}]
  (update-in
   m
   [scope]
   (fn [scope-map]
     (-> scope-map
         (assoc :bound-elements (conj (or (:bound-elements scope-map) #{}) element))
         (assoc :atom-ref atom-ref)
         (assoc :rendered-data @atom-ref)))))

(defn do-bindings!
  "Updates dom elements with their bound values, and returns a map of scopes
to their corresponding atoms, bound elements, and rendered state."
  ([]
     (do-bindings! js/document))
  ([dom]
     (let [elements (.querySelectorAll dom (str "[" bind-scope "]"))
           results (map -bind! elements)
           scope-bindings (reduce update-scopes {} results)
           _ (doall scope-bindings)] ;; Force evaluation
       scope-bindings)))

(defn- update-render-state
  [[scope-name {:keys [atom-ref rendered-data] :as scope-data} :as data-as-is]]
  (if (identical? @atom-ref rendered-data)
    data-as-is
    [scope-name (assoc scope-data :rendered-data @atom-ref)]))

(defn render-updated!
  "Calls render scopes that have rendered-data != current-bind-state"
  [m]
  (let [needs-rendering? (fn [[_ {:keys [atom-ref rendered-data]}]]
                            (not (identical? @atom-ref rendered-data)))
        scopes-that-need-rendering (map second (filter needs-rendering? m))]
    (if (seq scopes-that-need-rendering)
      (do
        (doseq [{:keys [atom-ref bound-elements]} scopes-that-need-rendering]
          (doseq [el bound-elements]
            (-render-all! el atom-ref)))
        (into {} (map update-render-state m))) ;; TODO Undoubtedly sub-optimal. But it'll do for now.
      m)))

(defn ^:export start
  ([]
     (start js/document))
  ([el]
     (let [scope-data (atom (do-bindings! el))]
       (js/setInterval #(swap! scope-data render-updated!) BINJ-LOOP-PERIOD))))
