(ns atombind.test
  (:require [goog.dom :as d]
            [atombind.core :as core]
            [goog.events :as events]
            [goog.testing.events :as te]
            [goog.style :as st]
            [goog.events.EventType :as EventType])
  (:import (goog.testing.events Event))
  (:require-macros [atombind.cljs-macros :refer [deftest]]))

(enable-console-print!)

(defn set-canvas
  [canvas]
  (set! (.-innerHTML (d/getElement "canvas")) canvas))

(def ^:export dog (atom {:name "Fido"}))

(deftest test-one-way-bind
  "One-way data binding"
  (set-canvas "<div id='one-way' bind-scope='atombind.test.dog'><span bind='name'></span></div>")
  (core/do-bindings!)
  (let [el (d/getElement "one-way")]
    (assert (= "Fido" (d/getTextContent (.querySelector el "span")))
            (str "Actual content: " (d/getTextContent el)))))

(deftest test-one-way-bind-text-input
  "One-way data binding with a text input"
  (set-canvas "<div id='one-way' bind-scope='atombind.test.dog'><input bind='name'></div>")
  (core/do-bindings!)
  (let [el (d/getElement "one-way")
        input (.querySelector el "input")]
    (assert (= "Fido" (.-value input)))))

(def ^:export scope-test (atom {}))

(deftest scope-binding
  "Checks returned scope object when binding"
  (set-canvas "
<div id='scope-test' bind-scope='atombind.test.scope_test'>
</div>")
  (let [result (core/do-bindings!)
        el (d/getElement "scope-test")]
    (assert (= {"atombind.test.scope_test" {:atom-ref scope-test
                                            :rendered-data @scope-test
                                            :bound-elements #{el}}}
               result)
            (str "Actual: " result))))

(def ^:export update-scope-test-scope (atom {}))

(defprotocol IMockRenderable
  (-was-rendered? [x]))

(defn mock-renderable
  []
  (let [rendered? (atom false)]
    (reify
      core/IRenderable
      (-render!
        [_ _]
        (reset! rendered? true))
      (-render-all!
        [_ _]
        (reset! rendered? true))
      IMockRenderable
      (-was-rendered?
        [_]
        @rendered?))))

(deftest render-updated-call-render
  "render-updated! should call -render! on bound elements for updated scope"
  (let [rndr1 (mock-renderable)
        rndr2 (mock-renderable)
        atom1 (atom {:foo "UPDATED"})
        atom2 (atom {:foo "bar"})
        scopes {"updated-scope" {:atom-ref atom1
                                 :rendered-data {:foo "bar"}
                                 :bound-elements #{rndr1}}
                "not-updated-scope" {:atom-ref atom2
                                     :rendered-data @atom2
                                     :bound-elements #{rndr2}}}
        result (core/render-updated! scopes)]
    (assert (= (@atom1 (get-in result ["updated-scope" :rendered-data]))))
    (assert (-was-rendered? rndr1))
    (assert (not (-was-rendered? rndr2)))))

(deftest render-updated-return-val
  "render-updated! should return the updated map"
  (let [atom1 (atom {:foo "bar"})
        rndr1 (mock-renderable)
        scopes {"scope" {:atom-ref atom1
                         :rendered-data {:foo "baz"}
                         :bound-elements #{rndr1}}}
        result-change (core/render-updated! scopes)
        result-no-change (core/render-updated! (assoc-in scopes
                                                         ["scope" :rendered-data]
                                                         @atom1))]
    (assert (not (nil? result-change)))
    (assert (not (nil? result-no-change)))))

(def ^:export show-test (atom {:editing false}))

(defn- visible?
  [el]
  (st/isElementShown el))

(deftest test-bind-show
  "Elements with a bind-show mapping to a true/false value"
  (set-canvas "
<div id='bind-show-test' bind-scope='atombind.test.show_test'>
<span id='text' bind-show='!editing'>Visible when NOT editing</span>
<span id='edit-panel' bind-show='editing'>
  <input name='foo'>
</span>
</div>")
  (let [result (core/do-bindings!)
        text (d/getElement "text")
        edit-panel (d/getElement "edit-panel")]
    (assert (visible? text))
    (assert (not (visible? edit-panel)))
    (swap! show-test assoc :editing true)
    (core/render-updated! result)
    (assert (not (visible? text)))
    (assert (visible? edit-panel))))


(def ^:export bind-func-test (atom {:value 123
                                    :increment (fn [ev]
                                                 (swap! bind-func-test
                                                        (fn [m]
                                                          (-> m 
                                                              (assoc :value (inc (:value m)))
                                                              (assoc :last-event ev)))))
                                    :last-event nil}))

(defn- fire-event
  "Fires a browser event. Returns the event that was fired."
  [target event-type]
  (let [ev (Event. event-type target)]
    (te/fireBrowserEvent ev)))

(deftest test-bind-func
  "Elements which have bind-func mappings"
  (set-canvas "
<div id='bind-func-test' bind-scope='atombind.test.bind_func_test'>
<span id='click-me' bind-func='click|increment'>
</span>
</div>")
  (swap! bind-func-test assoc :value 123)
  (let [result (core/do-bindings!)
        el (d/getElement "click-me")
        ev (fire-event el "click")]
    (assert (= 124 (:value @bind-func-test)) (str "It was " (:value @bind-func-test)))
    (assert (not (nil? (:last-event @bind-func-test))))))



(def ^:export attr-binding (atom {:my-class "big"}))

(deftest attribute-binding
  "Bind attribute"
  (set-canvas "
<div id='attr-binding'>
<div bind-scope='atombind.test.attr_binding'>
<span id='test-target' bind-attr='class|my-class' class='small'></span>
</div>
</div>")
  (let [result (core/do-bindings!)
        el (d/getElement "test-target")]
    (assert (= "big" (.getAttribute el "class")))))

(def ^:export attr-binding-func (atom {:attrfn (fn [el]
                                                 "abc 123")}))

(deftest attribute-func-binding
  "Bind attribute to function"
  (set-canvas "
<div id='attr-func-binding'>
<div bind-scope='atombind.test.attr_binding_func'>
<span id='test-target' bind-attr='foo|attrfn'></span>
</div>
</div>")
  (let [result (core/do-bindings!)
        el (d/getElement "test-target")]
    (assert (= "abc 123" (.getAttribute el "foo")))))

(defn- todo
  [id text]
  {:id id
   :text text})

(defn- default-todos
  []
  {:todos [(todo 0 "Buy milk")
           (todo 1 "Buy coffee")
           (todo 2 "Exercise")]})

(def ^:export todos (atom nil))

(def ^:private collection-html "
<div id='collection-test' bind-scope='atombind.test.todos'>
<ul id='todo-list'>
<li bind-each='todos' bind-attr='name|id'><span bind='text'></span></li>
</ul>
</div>")

(deftest test-collection-binding
  "Render collection bind and collection attribute binding"
  (reset! todos (default-todos))
  (set-canvas collection-html)
  (let [_ (core/do-bindings!)
        ul (d/getElement "todo-list")
        has? (fn [qry]
               (not (nil? (seq (js->clj (.querySelectorAll ul qry))))))
        get-text (fn [n]
                   (let [li (.querySelector ul (str "li[name='" n "']"))
                         span (.querySelector li "span")]
                     (d/getTextContent span)))
        li-count (count (.querySelectorAll ul "li"))]
    (assert (= 3 li-count) (str "Actual count was " li-count))
    (assert (and (has? "li[name='0']")
                 (has? "li[name='1']")
                 (has? "li[name='2']")))
    (assert (= "Buy milk" (get-text 0)))
    (assert (= "Buy coffee" (get-text 1)))
    (assert (= "Exercise" (get-text 2)))))


(deftest collection-binding-re-render
  "Re-render collection bind"
  (reset! todos (default-todos))
  (set-canvas collection-html)
  (let [scopes (core/do-bindings!)
        _ (swap! todos update-in [:todos] conj (todo 3 "New Todo!"))
        _ (core/render-updated! scopes)
        ul (d/getElement "todo-list")
        new-count (count (get @todos :todos))
        children-count (count (.querySelectorAll ul "li"))]
    (assert (= new-count children-count)
            (str "Actual count was " children-count " (expected " new-count ")"))))

(defn- toggle-done
  [data-atom id]
  (let [set-done (fn [{:keys [done] :as todo}]
                   (if (= id (:id todo))
                     (assoc todo :done (not done))
                     todo))]
    (swap! data-atom update-in [:todos] (comp vec (partial map set-done)))))

(def ^:export collection-func (atom {:todos [{:id 0
                                              :done false
                                              :text "Pick up the milk"
                                              :toggle-done #(toggle-done collection-func 0)}]}))

(deftest function-binding-in-collection
  "Function binding for a collection-bound dom element"
  (set-canvas "
<div bind-scope='atombind.test.collection_func'>
<ul>
<li bind-each='todos' bind-attr='todo|id'>
<span bind='text' bind-func='click|toggle-done'></span>
</li>
</ul>
</div>")
  (let [scopes (core/do-bindings!)
        li (.querySelector js/document "li[todo='0']")]
    (assert (not (nil? li)))
    (fire-event (.querySelector li "span") "click")
    (assert (get-in @collection-func [:todos 0 :done]))))

(def ctr (atom 0))

(def ^:export collection-scope-test (atom {:theText "This text should not appear"
                                           :a-func (fn [] (swap! ctr inc))
                                           :todos [{:id 0
                                                    :a-func (fn [])}]}))

(deftest collection-binds-are-ignored-when-outer-scope-is-bound
  "Bindings inside a 'bind-each' are ignored when the containing scope is being binded"
  (set-canvas "
<div bind-scope='atombind.test.collection_scope_test'>
<ul>
<li bind-each='todos'><span bind='theText' bind-func='click|a-func'>Hello</span></li>
</ul>
</div>")
  (let [_ (core/do-bindings!)
        li (.querySelector js/document "li")
        text (d/getTextContent li)
        _ (fire-event li "click")]
    (assert (not (= (:theText @collection-scope-test) text)))
    (assert (= 0 @ctr))))

(def ^:export coll-func-test-2 (atom {:ctr 0
                                      :coll [{:do-it (fn []
                                                       (swap! coll-func-test-2 update-in [:ctr] inc))}]}))

(deftest collection-function-bind-works-after-re-render
  "Function binding on collection works after re-render"
  (set-canvas "<div bind-scope='atombind.test.coll_func_test_2'>
<ul>
<li bind-each='coll'>
<button bind-func='click|do-it'>Click me</button>
</li>
</ul>
</div>")
  (let [scopes (core/do-bindings!)
        click #(fire-event (-> js/document
                               (.querySelector "li")
                               (.querySelector "button"))
                           "click")
        _ (click)
        _ (core/render-updated! scopes)
        _ (click)
        new-count (:ctr @coll-func-test-2)]
    (assert (= 2 new-count) (str "Count was " new-count))))

(deftest str-to-mappings-test
  "Test functionality of str-to-mappings"
  (let [test= (fn [expected input]
                (let [result (core/str-to-mappings input)]
                  (assert (= expected result)
                          (str "Got: " result))))]
    (test= [["foo" "bar"] ["baz" "bing"]]
           "foo|bar baz|bing")
    (test= [["foo" "{{keep it gangsta}}"]]
           "foo|{{keep it gangsta}}")
    (test= [["foo" "{{keep it gangsta}}"] ["bar" "baz"]]
           "foo|{{keep it gangsta}} bar|baz")))

(deftest js-eval-test
  "Test evaluation functionality"
  (let [data {:foo true
              :bar 1
              :baz "lerman"
              :kebab-case "kebabs-bro"
              :nested {:data "ftw"}}
        test= (fn [expected input]
                (let [result (core/evaluate data input)]
                  (assert (= expected result)
                          (str "Got: " result))))]
    (test= true "foo")
    (test= 1 "bar")
    (test= "lerman" "baz")
    (test= "dog" "{{%foo% ? 'dog' : 'poop'}}")
    (test= "larman" "{{%baz%.replace('e','a')}}")
    (test= 1 "{{%foo% ? %bar% : %baz%}}")
    (test= "lerman" "{{!%foo% ? %bar% : %baz%}}")
    (test= "kebabs-bro" "{{%kebab-case%}}")
    (test= "ftw" "{{%nested data%}}")))

(def ^:export bind-attr-js (atom {:done true}))

(deftest bind-attr-with-js-eval
  "Test binding an attribute using JS evaluation"
  (set-canvas "<div bind-scope='atombind.test.bind_attr_js'>
<span id='todo' bind-attr='class|{{%done% ? \"todo-done\" : \"todo\"}}'></span>
</div>")
  (let [_ (core/do-bindings!)
        span (d/getElement "todo")]
    (assert (= "todo-done" (.getAttribute span "class")))))

(def msgs (atom []))

(deftest bogus-bind-scope
  "Bogus scope gives us a useful error message"
  (set-canvas "<div bind-scope='bogus'>
</div>")
  (reset! msgs [])
  (with-redefs [core/err (fn [s]
                           (swap! msgs conj s))]
    (core/do-bindings!)
    (assert (= 1 (count @msgs)))))

(deftest invalid-js
  "Attempting to evaluate invalid JS causes error message"
  (reset! msgs [])
  (let [data {:hello "world"}
        input "{{bob}}"
        result (with-redefs [core/err (fn [s]
                           (swap! msgs conj s))]
                 (core/evaluate data input))]
    (assert (nil? result))
    (assert (= 1 (count @msgs)))))

(def ^:export zero-elems (atom {:todos []}))

(deftest collection-bind-zero-elements
  "Bind a collection with zero elements"
  (set-canvas "<div bind-scope='atombind.test.zero_elems'>
<ul>
<li bind-each='todos'></li>
</ul>
</div>")
  (let [lis (.querySelectorAll js/document "li")
        _ (core/do-bindings!)]
    (assert (= 1 (count lis)))
    (assert (not (visible? (first lis))))))

(def ^:export zero-elems-then-more (atom {:todos []}))

(deftest collection-bind-after-zero-elements
  "Bind a collection with zero elements, then add two"
  (set-canvas "<div bind-scope='atombind.test.zero_elems_then_more'>
<ul>
<li bind-each='todos' bind-attr='id|id'>
<span bind='text'></span>
</li>
</ul>
</div>")
  (swap! zero-elems-then-more assoc :todos [])
  (let [scopes (core/do-bindings!)
        _ (swap! zero-elems-then-more assoc :todos [{:text "hello" :id "todo1"}
                                                    {:text "world" :id "todo2"}])
        _ (core/render-updated! scopes)
        lis (.querySelectorAll js/document "li")
        todo1 (d/getElement "todo1")
        todo2 (d/getElement "todo2")]
    (assert (= 2 (count lis)) (str "Count was " (count lis)))
    (assert (visible? todo1))
    (assert (visible? todo2))))

(def ^:export nested (atom {:item {:price "$12.42"}}))

(deftest nested-bind
  "Test a bind to a nested value (a map within a map)"
  (set-canvas "<div bind-scope='atombind.test.nested'>
<span id='price' bind='item price'></span>
</div>")
  (let [_ (core/do-bindings!)
        span (d/getElement "price")]
    (assert (= "$12.42" (d/getTextContent span)))))

(def ^:export two-way (atom {:price "12.32"}))

(deftest two-way-bind
  "Test two way binding"
  (set-canvas "<div bind-scope='atombind.test.two_way'>
<input id='priceInput' type='text' bind='price'>
</div>")
  (let [_ (core/do-bindings!)
        input (d/getElement "priceInput")]
    (set! (.-value input) "500.00")
    (fire-event input EventType/CHANGE))
  (assert (= "500.00" (get-in @two-way [:price]))))

(def ^:export nested-two-way (atom {:item {:price "12.32"}}))

(deftest nested-two-way-bind
  "Test nested two way binding"
  (set-canvas "<div bind-scope='atombind.test.nested_two_way'>
<input id='priceInput' type='text' bind='item price'>
</div>")
  (let [_ (core/do-bindings!)
        input (d/getElement "priceInput")]
    (set! (.-value input) "500.00")
    (fire-event input EventType/CHANGE))
  (assert (= "500.00" (get-in @nested-two-way [:item :price]))))

(defn ^:export run-tests
  []
  (run-all-tests))
