(ns atombind.test
  (:require [goog.dom :as d]
            [atombind.scopes :as s]
            [atombind.core :as core]
            [atombind.logging :as l]
            [goog.events :as events]
            [goog.testing.events :as te]
            [goog.style :as st]
            [goog.events.EventType :as EventType]
            [atombind.testutil :refer [set-canvas]])
  (:import (goog.testing.events Event))
  (:require-macros [atombind.cljs-macros :refer [deftest]]))

(enable-console-print!)

(def ^:export dog (atom {:name "Fido"}))

(deftest test-one-way-bind
  "One-way data binding"
  (set-canvas "<div id='one-way' bind-scope='atombind.test.dog'><span bind='name'></span></div>")
  (core/update-dom!)
  (let [el (d/getElement "one-way")]
    (assert (= "Fido" (d/getTextContent (.querySelector el "span")))
            (str "Actual content: " (d/getTextContent el)))))

(deftest test-one-way-bind-with-js-eval
  "One-way data binding with js eval"
  (set-canvas "<div id='one-way' bind-scope='atombind.test.dog'><span bind='{{%name% === \"Fido\" ? \"Right!\" : \"wrong...\"}}'></span></div>")
  (core/update-dom!)
  (let [el (d/getElement "one-way")]
    (assert (= "Right!" (d/getTextContent (.querySelector el "span")))
            (str "Actual content: " (d/getTextContent el)))))

(deftest test-one-way-bind-text-input
  "One-way data binding with a text input"
  (set-canvas "<div id='one-way' bind-scope='atombind.test.dog'><input bind='name'></div>")
  (core/update-dom!)
  (let [el (d/getElement "one-way")
        input (.querySelector el "input")]
    (assert (= "Fido" (.-value input)))))

(deftest test-text-input-complex-binding
  "Text input should not allow complex bindings. Error message should result."
  (set-canvas "<div id='one-way' bind-scope='atombind.test.dog'><input bind='{{%name%}}'></div>")
  (let [msgs (atom [])]
    (with-redefs [l/err (fn [s]
                             (swap! msgs conj s))]
      (core/update-dom!)
      (assert (= 1 (count @msgs))))))

(def ^:export update-scope-test-scope (atom {}))

(def ^:export show-test (atom {:editing false}))

(defn- visible?
  [el]
  (st/isElementShown el))

(def ^:private hidden? (complement visible?))

(deftest test-bind-show
  "Test bind-show functionality"
  (set-canvas "
<div id='bind-show-test' bind-scope='atombind.test.show_test'>
<span id='text' bind-show='{{!%editing%}}'>Visible when NOT editing</span>
<span id='edit-panel' bind-show='editing'>
  <input name='foo'>
</span>
</div>")
  (binding [s/*scopes* (atom {})]
    (let [_ (core/update-dom!)
          text (d/getElement "text")
          edit-panel (d/getElement "edit-panel")]
      (assert (visible? text))
      (assert (not (visible? edit-panel)))
      (swap! show-test assoc :editing true)
      (core/update-dom!)
      (assert (not (visible? text)))
      (assert (visible? edit-panel)))))


(def ^:export bind-func-test (atom {:value 123
                                    :increment (fn [ev]
                                                 (swap! bind-func-test
                                                        (fn [m]
                                                          (-> m 
                                                              (assoc :value (inc (:value m)))
                                                              (assoc :last-event ev)))))
                                    :last-event nil}))

(defn- fire-event
  "Fires a browser event. Returns the event that was fired."
  [target event-type]
  (let [ev (Event. event-type target)]
    (te/fireBrowserEvent ev)))

(deftest test-bind-func
  "Bind function with bind-func mappings"
  (set-canvas "
<div id='bind-func-test' bind-scope='atombind.test.bind_func_test'>
<span id='click-me' bind-func='click|increment'>
</span>
</div>")
  (swap! bind-func-test assoc :value 123)
  (let [_ (core/update-dom!)
        el (d/getElement "click-me")
        ev (fire-event el "click")]
    (assert (= 124 (:value @bind-func-test)) (str "It was " (:value @bind-func-test)))
    (assert (not (nil? (:last-event @bind-func-test))))))



(def ^:export attr-binding (atom {:my-class "big"}))

(deftest attribute-binding
  "Bind attribute"
  (set-canvas "
<div id='attr-binding'>
<div bind-scope='atombind.test.attr_binding'>
<span id='test-target' bind-attr='class|my-class' class='small'></span>
</div>
</div>")
  (let [result (core/update-dom!)
        el (d/getElement "test-target")]
    (assert (= "big" (.getAttribute el "class")))))

(def ^:export attr-binding-func (atom {:attrfn (fn [el]
                                                 "abc 123")}))

(deftest attribute-func-binding
  "Bind attribute to function"
  (set-canvas "
<div id='attr-func-binding'>
<div bind-scope='atombind.test.attr_binding_func'>
<span id='test-target' bind-attr='foo|attrfn'></span>
</div>
</div>")
  (let [result (core/update-dom!)
        el (d/getElement "test-target")]
    (assert (= "abc 123" (.getAttribute el "foo")))))

(defn- todo
  [id text]
  (atom {:id id
         :text text}))

(defn- default-todos
  []
  {:todos [(todo 0 "Buy milk")
           (todo 1 "Buy coffee")
           (todo 2 "Exercise")]})

(def ^:export todos (atom nil))

(def ^:private collection-html "
<div id='collection-test' bind-scope='atombind.test.todos'>
<ul id='todo-list'>
<li bind-each='todos' bind-attr='name|id'><span bind='text'></span></li>
</ul>
</div>")

(deftest test-collection-binding
  "Render collection bind and collection attribute binding"
  (reset! todos (default-todos))
  (set-canvas collection-html)
  (let [_ (core/update-dom!)
        ul (d/getElement "todo-list")
        has? (fn [qry]
               (not (nil? (seq (js->clj (.querySelectorAll ul qry))))))
        get-text (fn [n]
                   (let [li (.querySelector ul (str "li[name='" n "']"))
                         span (.querySelector li "span")]
                     (d/getTextContent span)))
        li-count (count (.querySelectorAll ul "li"))]
    (assert (= 3 li-count) (str "Actual count was " li-count))
    (assert (and (has? "li[name='0']")
                 (has? "li[name='1']")
                 (has? "li[name='2']")))
    (assert (= "Buy milk" (get-text 0)))
    (assert (= "Buy coffee" (get-text 1)))
    (assert (= "Exercise" (get-text 2)))))


(deftest collection-binding-re-render
  "Re-render collection bind"
  (reset! todos (default-todos))
  (set-canvas collection-html)
  (binding [s/*scopes* (atom {})]
    (core/update-dom!)
    (swap! todos update-in [:todos] conj (todo 3 "New Todo!"))
    (core/update-dom!)
    (let [ul (d/getElement "todo-list")
          new-count (count (get @todos :todos))
          children-count (count (.querySelectorAll ul "li"))]
      (assert (= new-count children-count)
              (str "Actual count was " children-count " (expected " new-count ")")))))

(defn- toggle-done
  [data-atom id]
  (let [set-done (fn [{:keys [done] :as todo}]
                   (if (= id (:id todo))
                     (assoc todo :done (not done))
                     todo))]
    (swap! data-atom update-in [:todos] (comp vec (partial map set-done)))))

(def ^:export collection-func (atom {:todos [(atom {:id 0
                                                    :done false
                                                    :text "Pick up the milk"
                                                    :toggle-done (fn [a]
                                                                   (swap! a assoc :done true))})]}))

(deftest function-binding-in-collection
  "Function binding for a collection-bound dom element"
  (set-canvas "
<div bind-scope='atombind.test.collection_func'>
<ul>
<li bind-each='todos' bind-attr='todo|id'>
<span bind='text' bind-func='click|toggle-done'></span>
</li>
</ul>
</div>")
  (let [scopes (core/update-dom!)
        li (.querySelector js/document "li[todo='0']")]
    (assert (not (nil? li)))
    (fire-event (.querySelector li "span") "click")
    (assert (:done @(get-in @collection-func [:todos 0])))))

(def ctr (atom 0))

(def ^:export collection-scope-test (atom {:theText "This text should not appear"
                                           :a-func (fn [] (swap! ctr inc))
                                           :todos [(atom {:id 0
                                                          :a-func (fn [])})]}))

(deftest collection-binds-are-ignored-when-outer-scope-is-bound
  "Bindings inside a 'bind-each' are ignored when the containing scope is being binded"
  (set-canvas "
<div bind-scope='atombind.test.collection_scope_test'>
<ul>
<li bind-each='todos'><span bind='theText' bind-func='click|a-func'>Hello</span></li>
</ul>
</div>")
  (let [_ (core/update-dom!)
        li (.querySelector js/document "li")
        text (d/getTextContent li)
        _ (fire-event li "click")]
    (assert (not (= (:theText @collection-scope-test) text)))
    (assert (= 0 @ctr))))

(def ^:export coll-func-test-2 (atom {:ctr 0
                                      :coll [(atom {:do-it (fn []
                                                             (swap! coll-func-test-2 update-in [:ctr] inc))})]}))

(deftest collection-function-bind-works-after-re-render
  "Function binding on collection works after re-render"
  (set-canvas "<div bind-scope='atombind.test.coll_func_test_2'>
<ul>
<li bind-each='coll'>
<button bind-func='click|do-it'>Click me</button>
</li>
</ul>
</div>")
  (let [scopes (core/update-dom!)
        click #(fire-event (-> js/document
                               (.querySelector "li")
                               (.querySelector "button"))
                           "click")
        _ (click)
        _ (core/update-dom!)
        _ (click)
        new-count (:ctr @coll-func-test-2)]
    (assert (= 2 new-count) (str "Count was " new-count))))

(deftest str-to-mappings-test
  "Test functionality of str-to-mappings"
  (let [test= (fn [expected input]
                (let [result (core/str-to-mappings input)]
                  (assert (= expected result)
                          (str "Got: " result))))]
    (test= [["foo" "bar"] ["baz" "bing"]]
           "foo|bar baz|bing")
    (test= [["foo" "{{keep it gangsta}}"]]
           "foo|{{keep it gangsta}}")
    (test= [["foo" "{{keep it gangsta}}"] ["bar" "baz"]]
           "foo|{{keep it gangsta}} bar|baz")))

(deftest js-eval-test
  "Test evaluation functionality"
  (let [data {:foo true
              :bar 1
              :baz "lerman"
              :kebab-case "kebabs-bro"
              :nested {:data "ftw"}}
        test= (fn [expected input]
                (let [result (core/evaluate data input)]
                  (assert (= expected result)
                          (str "Got: " result))))]
    (test= true "foo")
    (test= 1 "bar")
    (test= "lerman" "baz")
    (test= "dog" "{{%foo% ? 'dog' : 'poop'}}")
    (test= "larman" "{{%baz%.replace('e','a')}}")
    (test= 1 "{{%foo% ? %bar% : %baz%}}")
    (test= "lerman" "{{!%foo% ? %bar% : %baz%}}")
    (test= "kebabs-bro" "{{%kebab-case%}}")
    (test= "ftw" "{{%nested data%}}")))

(def ^:export bind-attr-js (atom {:done true}))

(deftest bind-attr-with-js-eval
  "Test binding an attribute using JS evaluation"
  (set-canvas "<div bind-scope='atombind.test.bind_attr_js'>
<span id='todo' bind-attr='class|{{%done% ? \"todo-done\" : \"todo\"}}'></span>
</div>")
  (let [_ (core/update-dom!)
        span (d/getElement "todo")]
    (assert (= "todo-done" (.getAttribute span "class")))))

(def msgs (atom []))

(deftest bogus-bind-scope
  "Bogus scope gives us a useful error message"
  (set-canvas "<div bind-scope='bogus'>
</div>")
  (reset! msgs [])
  (with-redefs [l/err (fn [s]
                        (swap! msgs conj s))]
    (core/update-dom!)
    (assert (not (empty? @msgs)))))

(deftest invalid-js
  "Attempting to evaluate invalid JS causes error message"
  (reset! msgs [])
  (let [data {:hello "world"}
        input "{{bob}}"
        result (with-redefs [l/err (fn [s]
                           (swap! msgs conj s))]
                 (core/evaluate data input))]
    (assert (nil? result))
    (assert (= 1 (count @msgs)))))

(def ^:export zero-elems (atom {:todos []}))

(deftest collection-bind-zero-elements
  "Bind a collection with zero elements"
  (set-canvas "<div bind-scope='atombind.test.zero_elems'>
<ul>
<li bind-each='todos'></li>
</ul>
</div>")
  (let [lis (.querySelectorAll js/document "li")
        _ (core/update-dom!)]
    (assert (= 1 (count lis)))
    (assert (not (visible? (first lis))))))

(def ^:export zero-elems-then-more (atom {:todos []}))

(deftest collection-bind-after-zero-elements
  "Bind a collection with zero elements, then add two"
  (set-canvas "<div bind-scope='atombind.test.zero_elems_then_more'>
<ul>
<li bind-each='todos' bind-attr='id|id'>
<span bind='text'></span>
</li>
</ul>
</div>")
  (binding [s/*scopes* (atom {})]
    (swap! zero-elems-then-more assoc :todos [])
    (core/update-dom!)
    (swap! zero-elems-then-more assoc :todos [(atom {:text "hello" :id "todo1"})
                                              (atom {:text "world" :id "todo2"})])
    (core/update-dom!)
    (let [lis (.querySelectorAll js/document "li")
          todo1 (d/getElement "todo1")
          todo2 (d/getElement "todo2")]
      (assert (= 2 (count lis)) (str "Count was " (count lis)))
      (assert (visible? todo1))
      (assert (visible? todo2)))))

(def ^:export nested (atom {:item {:price "$12.42"}}))

(deftest nested-bind
  "Test a bind to a nested value (a map within a map)"
  (set-canvas "<div bind-scope='atombind.test.nested'>
<span id='price' bind='item price'></span>
</div>")
  (let [_ (core/update-dom!)
        span (d/getElement "price")]
    (assert (= "$12.42" (d/getTextContent span)))))

(def ^:export two-way (atom {:price "12.32"}))

(deftest two-way-bind
  "Test two way binding"
  (set-canvas "<div bind-scope='atombind.test.two_way'>
<input id='priceInput' type='text' bind='price'>
</div>")
  (let [_ (core/update-dom!)
        input (d/getElement "priceInput")]
    (set! (.-value input) "500.00")
    (fire-event input EventType/CHANGE))
  (assert (= "500.00" (get-in @two-way [:price]))))

(def ^:export nested-two-way (atom {:item {:price "12.32"}}))

(deftest nested-two-way-bind
  "Test nested two way binding"
  (set-canvas "<div bind-scope='atombind.test.nested_two_way'>
<input id='priceInput' type='text' bind='item price'>
</div>")
  (let [_ (core/update-dom!)
        input (d/getElement "priceInput")]
    (set! (.-value input) "500.00")
    (fire-event input EventType/CHANGE))
  (assert (= "500.00" (get-in @nested-two-way [:item :price]))))

(def ^:export collection-editing (atom {:todos [(atom {:text "Hello"
                                                       :id "1"})
                                                (atom {:text "World"
                                                       :id "2"})]}))

(deftest edit-inside-collection
  "Test editing functionality inside a collection"
  (set-canvas "
<div bind-scope='atombind.test.collection_editing'>
<ul>
<li bind-each='todos'>
  <input bind='text' bind-attr='id|{{\"todo-\" + %id%}}'>
  <span bind='text'></span>
</li>
</ul>
</div>")
  (let [_ (core/update-dom!)
        input (d/getElement "todo-1")]
    (set! (.-value input) "New Text")
    (fire-event input EventType/CHANGE)
    (assert (= "New Text" (:text @(get-in @collection-editing [:todos 0])))
            (str "Actual value was: " (:text @(get-in @collection-editing [:todos 0]))))))

(def ^:export more-coll-rendering (atom {:items [(atom {:id 0
                                                        :name "Art"
                                                        :price "29.99"
                                                        :editing false})
                                                 (atom {:id 1
                                                        :name "Furniture"
                                                        :price "299.99"
                                                        :editing false})]}))

(deftest re-render-individual-item-in-collection
  "Test re-rendering an individual collection item"
  (set-canvas "
<div bind-scope='atombind.test.more_coll_rendering'>
<ul>
<li bind-each='items'>
  <span bind='name'></span> - <span bind='price' bind-attr='id|id'></span>
  <div bind-attr='id|{{\"edit-pane-\" + %id%}}' bind-show='editing'>Edit Panel</div>
</li>
</ul>
</div>")
  (binding [s/*scopes* (atom {})]
    (core/update-dom!)
    (let [price-el (d/getElement "0")
          item (get-in @more-coll-rendering [:items 0])]
      (assert (= "29.99" (d/getTextContent price-el)))
      (assert (= "299.99" (d/getTextContent (d/getElement "1"))))
      (assert (not (visible? (d/getElement "edit-pane-0"))))
      (swap! item assoc :price "15.00")
      (core/update-dom!)
      (assert (= "15.00" (d/getTextContent price-el)))
      (swap! item assoc :editing true)
      (core/update-dom!)
      (assert (visible? (d/getElement "edit-pane-0"))))))

(def ^:export bind-show-scope (atom {:foo "bar"}))

(deftest bind-show-with-complex
  "Using a 'bind-show' with a complex evaluation"
  (set-canvas "<div>
<span bind-scope='atombind.test.bind_show_scope'>
  <span id='should-be-visible' bind-show='{{%foo% === \"bar\"}}'>Hello</span>
  <span id='should-be-hidden' bind-show='{{%foo% !== \"bar\"}}'>World</span>
</span>
</div>")
  (core/update-dom!)
  (let [shown (d/getElement "should-be-visible")
        hidden (d/getElement "should-be-hidden")]
    (assert (visible? shown))
    (assert (not (visible? hidden)))))

(deftest bind-hide
  "Using bind-hide should give us the inverse of bind-show"
  (set-canvas "<div>
<span bind-scope='atombind.test.bind_show_scope'>
<span id='hidden' bind-hide='foo'>I am hidden</span>
<span id='shown' bind-show='foo'>I am showing!</span>
</span>
</div>")
  (core/update-dom!)
  (assert (visible? (d/getElement "shown")))
  (assert (hidden? (d/getElement "hidden"))))

(def ^:export a-collection (atom {:text "outer text"
                                  :items [(atom {:text "item text 1"})
                                          (atom {:text "item text 2"})]}))

(deftest collection-bind-with-overlapping-values
  "Binding collection items with overlapping values"
  (set-canvas "<div>
<div bind-scope='atombind.test.a_collection'>
<ul id='the-items'>
<li bind-each='items'><span bind='text'></span></li>
</ul>
</div>")
  (core/update-dom!)
  (let [ul (d/getElement "the-items")
        lis (d/getChildren ul)
        last-child (last lis)
        last-child-span (.querySelector last-child "span")]
    (assert (= 2 (count lis)))
    (assert (= "item text 2" (d/getTextContent last-child-span)))))

;; TODO Better re-rendering of collections (calculate a diff)

(defn ^:export run-tests
  []
  (println "** CORE TESTS **")
  (run-all-tests))
