(ns atombind.core
  (:require [atombind.constants :as c]
            [atombind.scopes :as s]
            [atombind.logging :as l]
            [goog.dom :as d]
            [goog.array :as a]
            [goog.dom.classlist :as cl]
            [goog.events :as e]
            [goog.style :as st]
            [goog.events.EventType :as EventType]))

(defprotocol IRenderable
  (-render! [el data-atom] "Render an individual element")
  (-render-all! [el data-atom] "Renders all sub-elements")
  (-rendered? [el] "Has the given element been (-render!)'d at least once?")
  (-set-attrs! [el data-atom] "Sets attributes according to data-binding")
  (-render-each! [el data-atom] "Repeats the element for each item in the bound collection"))

(defprotocol IHideable
  (-hide-or-show! [el data-atom] "Hides an element if it's 'bind-show' evaluates to false."))

(defprotocol IStringExt
  (-starts-with? [str prefix]))

(extend-type string
  IStringExt
  (-starts-with? [str prefix]
    "Taken from http://stackoverflow.com/questions/646628/how-to-check-if-a-string-startswith-another-string"
    (identical? 0 (.lastIndexOf str prefix 0))))

(extend-type js/NodeList
  ISeqable
  (-seq [nl] (array-seq nl)))

(defn- str-to-mappings
  [str]
  (let [split-regex #"([\w-_]+\|[\w-_]+|[\w-_]+\|\{\{.*?\}\})\s+"
        blank? (fn [s] (= (.trim s) ""))]
    (->> (js->clj (.split str split-regex))
         (filter (complement blank?))
         (map #(js->clj (.split % "|"))))))

(defn- to-map-key
  "Turns 'foo bar baz' into (:foo :bar :baz)"
  [s]
  (map keyword (-> s
                   (.split (js/RegExp. " +"))
                   js->clj)))

(def ^:private complex-regex-expr #"^\{\{(.*?)\}\}$")

(defn- get-complex-binding
  [s]
  (.match s complex-regex-expr))

(defn- is-complex-binding?
  [s]
  (not (nil? (get-complex-binding s))))

(defn- evaluate
  [data binding]
  (let [complex-binding (get-complex-binding binding)]
    (if complex-binding
      (let [inner (aget complex-binding 1)
            inner-binding-regex (js/RegExp. "\\%([\\w_ -]+?)\\%" "g")
            inner-bindings (.match inner inner-binding-regex)
            new-inner (reduce (fn [s inner-binding]
                                (let [expr (.replace inner-binding
                                                     (js/RegExp. "\\%" "g")
                                                     "")
                                      evald (evaluate data expr)
                                      string? (= (type evald) js/String)]
                                  (.replace s
                                            (js/RegExp. inner-binding "g")
                                            (if string?
                                              (str "\"" evald "\"")
                                              evald))))
                              inner
                              inner-bindings)
            js-str (str "(function(){return " new-inner ";})();")]
        (try
          (js/eval js-str)
          (catch js/Error e
            (l/err (str "Error evaluating statement: "
                      new-inner
                      "\nError: "
                      e))
            nil)))
      (get-in data (to-map-key binding)))))

(defn- add-change-handler!
  [el data-atom]
  (when-not (cl/contains el (:change-handler c/CLASSES))
    (cl/add el (:change-handler c/CLASSES))
    (e/listen el
              (array EventType/CHANGE EventType/KEYUP EventType/INPUT)
              (fn [_]
                (when-not (cl/contains el (:dirty c/CLASSES))
                  (cl/add el (:dirty c/CLASSES))
                  (let [bind-key (to-map-key (.getAttribute el c/bind))]
                    (swap! data-atom assoc-in bind-key (.-value el)))
                  (js/setTimeout (fn [] (cl/remove el (:dirty c/CLASSES))) c/ATOMBIND-LOOP-PERIOD))))))

(defn- bind-event-handlers!
  [el data-atom]
  (when-not (cl/contains el (:event-handler c/CLASSES))
    (cl/add el (:event-handler c/CLASSES))
    (let [binding-string (.getAttribute el c/bind-func)
          bindings (str-to-mappings binding-string)]
      (doseq [[event func-name] bindings]
        (e/listen el event (fn []
                             (let [func (evaluate @data-atom func-name)]
                               (func data-atom))))))))

(extend-type js/HTMLInputElement
  IRenderable
  (-render! [el data-atom]
    (when-not (cl/contains el (:dirty c/CLASSES))
      (let [binding (.getAttribute el c/bind)]
        (if (is-complex-binding? binding)
          (l/err (str "ERROR binding " binding ": HTML <input> elements cannot have {{complex bindings}} assigned to the 'bind' attribute. Please use a simple binding."))
          (when-let [content (->> binding
                                  to-map-key
                                  (get-in @data-atom))]
            (set! (.-value el) content)
            (add-change-handler! el data-atom)))))))

(extend-type js/HTMLTextAreaElement
  IRenderable
  (-render! [el data-atom]
    (when-not (cl/contains el (:dirty c/CLASSES))
      (let [binding (.getAttribute el c/bind)]
        (if (is-complex-binding? binding)
          (l/err (str "ERROR binding " binding ": HTML <textarea> elements cannot have {{complex bindings}} assigned to the 'bind' attribute. Please use a simple binding."))
          (when-let [content (->> binding
                                  to-map-key
                                  (get-in @data-atom))]
            (set! (.-value el) content)
            (add-change-handler! el data-atom)))))))

(defn- random
  []
  (.floor js/Math (.random js/Math 100)))

(defn- clean-element!
  "Removes all pseudo-classes and data that have been added to an element (and its children)."
  [el]
  (doseq [[_ class] c/CLASSES]
    (cl/remove el class))
  (doseq [e (d/getChildren el)]
    (clean-element! e)))

;; TODO Make collection re-rendering more efficient. Right now, every time the atom changes, the collections are re-rendered.
(defn- unrender-collections!
  [el]
  (let [els (.querySelectorAll el (str "[" c/bound-to-coll "]"))
        templates (.querySelectorAll el (str "[" c/bind-each "]"))]
    (doseq [x els]
      (d/removeNode x))
    (doseq [x templates]
      (clean-element! x))))

(extend-protocol ISeqable

  js/HTMLCollection
  (-seq [coll]
    (cond (= 0 (.-length coll)) nil
          :else (lazy-seq
                 (cons (aget coll 0) (-seq (a/slice coll 1))))))

  array
  (-seq [coll]
    (cond (= 0 (.-length coll)) nil
          :else (lazy-seq
                 (cons (aget coll 0) (-seq (a/slice coll 1)))))))

(defn- is-scope?
  [el]
  (let [has-attr? (fn
                    [s]
                    (not (nil? (.getAttribute el s))))]
    (or (has-attr? c/bind-scope)
        (has-attr? c/bind-each)
        (has-attr? c/scope-id)
        (has-attr? c/bound-to-coll))))

(def ^:private tmp (atom 0))

(defn- same-scope?
  [outer-elem inner-elem]
  "Returns whether or not inner-elem belongs in the same scope as outer elem."
  (loop [e inner-elem]
    (cond (nil? e) true
          (identical? e outer-elem) true
          (is-scope? e) false
          :else (recur (d/getParentElement e)))))

(extend-type js/HTMLElement
  IRenderable
  (-render! [el data-atom]
    (when-let [content (->> (.getAttribute el c/bind)
                            (evaluate @data-atom))]
      (d/setTextContent el content)))
  (-render-all! [el data-atom]
    (.setAttribute el c/rendered "true")
    (let [find-and-do (fn [attr-name f & [include-top-level? _]]
                        (doseq [e (.querySelectorAll el (str "[" attr-name "]"))]
                          (when (same-scope? el e)
                            (let [val (.getAttribute e attr-name)]
                              (when val
                                (f e data-atom)))))
                        (when include-top-level?
                          (let [val (.getAttribute el attr-name)]
                            (when val
                              (f el data-atom)))))
          render-collections! (fn []
                                (unrender-collections! el)
                                (doseq [e (.querySelectorAll el (str "[" c/bind-each "]"))]
                                  (-render-each! e data-atom)))]
      (render-collections!)
      (find-and-do c/bind-attr -set-attrs! true)
      (find-and-do c/bind -render!)
      (find-and-do c/bind-show -hide-or-show!)
      (find-and-do c/bind-hide -hide-or-show!)
      (find-and-do c/bind-func bind-event-handlers! true)))
  (-rendered? [el]
    (= (.getAttribute el c/rendered) "true"))
  (-set-attrs! [el data-atom]
    (let [attr-bindings (str-to-mappings (.getAttribute el c/bind-attr))]
      (doseq [[attr binding] attr-bindings]
        (let [value (evaluate @data-atom binding)]
          (when-let [to-set (if (= js/Function (type value))
                              (value el)
                              value)]
            (.setAttribute el attr to-set))))))
  (-render-each! [el data-atom]
    (st/setElementShown el true)
    (let [coll-ref (.getAttribute el c/bind-each)
          coll (get @data-atom (keyword coll-ref))
          template (let [x (.cloneNode el true)]
                     (.removeAttribute x c/bind-each)
                     (.setAttribute x c/bound-to-coll coll-ref)
                     x)]
      (when (seq coll)
        (loop [last-node nil
               first? (nil? last-node)
               new-node (.cloneNode el true)
               data (first coll)
               remaining-data (rest coll)]
          (s/-register-scope! data)
          (s/-init-item! new-node data)
          (-render-all! new-node data)
          (if first?
            (d/replaceNode new-node el)
            (d/insertSiblingAfter new-node last-node))
          (when (seq remaining-data)
            (recur new-node
                   false
                   (.cloneNode template true)
                   (first remaining-data)
                   (rest remaining-data)))))
      (when-not (seq coll)
        (st/setElementShown el false))))
  IHideable
  (-hide-or-show! [el data-atom]
    (let [show-expr (.getAttribute el c/bind-show)
          hide-expr (.getAttribute el c/bind-hide)
          expr (or show-expr hide-expr)
          result (evaluate @data-atom expr)
          result-bool (if result
                         true
                         false)
          should-show? (if hide-expr
                         (not result-bool)
                         result-bool)]
      (st/setElementShown el should-show?))))

(defn- render-updated
  [el]
  (when (s/-scopeable? el)
    (if (s/-scoped? el)
      (let [scope (s/-get-scope el)]
        (when (or (s/-updated? scope)
                  (not (-rendered? el)))
          (-render-all! el scope)))
      (do
        (let [scope (s/-get-scope! el)]
          (-render-all! el scope))))))

(defn- update-dom!
  []
  (let [updated-scopes (atom #{})
        top-level-scopes (.querySelectorAll js/document (str "[" c/bind-scope "]"))
        scope-id-elements (.querySelectorAll js/document (str "[" c/scope-id "]"))]
    (doseq [el (concat top-level-scopes scope-id-elements)]
      (render-updated el)
      (when-let [scope (s/-get-scope! el)]
        (when (s/-updated? scope)
          (swap! updated-scopes conj scope))))
    (doseq [s @updated-scopes]
      (s/-save-state s))))

(defn bind-scope!
  [element scope]
  (s/-init-item! element scope))

(defn ^:export start
  []
  (js/setInterval #(update-dom!) c/ATOMBIND-LOOP-PERIOD))
