(ns io.cvcf.frag.basic
  (:require
   [clojure.string :as str]
   [io.cvcf.frag.core :refer [render]]
   [io.cvcf.frag.css :refer [>css]]
   [io.cvcf.frag.u.datetime :refer [year-range]]
   [io.cvcf.frag.u.names :refer [>id >name named?]]
   [io.cvcf.frag.u.urls :refer [social]]))

(defmethod render :f/value
  [{:keys [value]}]
  (cond-> value
    (named? value) >name
    (map? value)   render
    :else          identity))

(defmethod render :f/url
  [{:keys [style href platform profile]}]
  (case style
    :email  (str "mailto:" href)
    :tel    (str "tel:" href)
    :social (social platform profile)
    href))

(defmethod render :comp/children
  [{:keys [children]}]
  (map render children))

(defmethod render :comp/box
  [{:keys [id children styles on] :as opts}]
  [:div {:id (>id id) :class (>css opts) :on on :style styles}
   (render {:type :comp/children :children children})])

(defmethod render :comp/empty
  [{:keys [id on] :as opts}]
  [:span {:id (>id id) :class (>css opts) :on on} ""])

(defmethod render :comp/break
  [_]
  [:br])

(defmethod render :comp/link
  [{:keys [id label on] :as opts}]
  (let [url (render (merge opts {:type :f/url}))]
    [:a {:id (>id id) :href url :class (>css opts) :on on}
     (render {:type :f/value :value label})]))

(defmethod render :comp/links
  [{:keys [id children on] :as opts}]
  [:nav {:id (>id id) :class (>css opts) :on on}
   (render {:type :comp/children :children children})])

(defmethod render :comp/copyright
  [{:keys [id on] :as opts}]
  [:span {:id (>id id) :class (>css opts) :on on}
   "Copyright © "
   (str/join "-" (year-range opts))
   ". All rights reserved."])

(defmethod render :comp/heading
  [{:keys [id importance text on] :as opts}]
  (let [heading #(keyword (str "h" %))]
    [(heading importance)
     {:id (>id id) :class (>css opts) :on on} text]))

(defmethod render :comp/address
  [{:keys [id street1 street2 city state zip country on]
    :as   opts
    :or   {street2 ""}}]
  [:address {:id (>id id) :class (>css opts) :on on}
   street1 " " street2 [:br]
   city ", " state " " zip [:br]
   country])

(defmethod render :comp/text
  [{:keys [id text sep on] :as opts}]
  (let [p (fn [id text]
            [:p {:id (>id id) :class (>css opts) :on on} text])]
    (cond
      (string? text) (p id text)
      (string? sep)  (p id (str/join sep text))
      (coll? sep)    (p id (interpose sep text))
      (coll? text)   (mapcat #(vec [(p (>id id %1) %2)])
                             (range)
                             text))))

(defmethod render :comp/social
  [{:keys [id platform profile on] :as opts}]
  (render
   {:type :comp/link
    :style :social
    :platform platform
    :profile profile
    :label {:type :comp/empty :id (>id id) :class (>css opts) :on on}}))

(defmethod render :comp/list
  [{:keys [id children on style] :as opts}]
  (let [tag (if (= style :ordered) :ol :ul)]
    [tag {:id (>id id) :class (>css opts) :on on}
     (render {:type :comp/children :children children})]))

(defmethod render :comp/list-item
  [{:keys [id on value] :as opts}]
  [:li {:id (>id id) :class (>css opts) :on on}
   (render {:type :f/value :value value})])
