(ns clj.new.io.dominic.juxt-wedge
  (:require
    [clj.new.templates :refer [renderer project-data ->files]]
    [io.dominic.wedge.impl.dev-tooling :as impl.dev-tooling]))

(defn- generate-port
  "Generates 'pretty' port sequences in the form of:
  xx00
  xyxy"
  []
  (let [base 20
        upper-limit 99
        dd-port (+ (rand-int (- (inc upper-limit) base)) base)]
    (rand-nth
      [(format "%d%d" dd-port dd-port)
       (format "%d00" dd-port)])))

(def supported-flags
  {"sass" {:description "Configure a sass builder for this app"}
   "cljs" {:description "Configure figwheel and cljs for this app"}
   "reframe" {:description "Configure skeleton reframe structure, implies --cljs"}})

(def flags-opts (set (map #(str "--" %) (conj (keys supported-flags) "no-web"))))

;;The following functions involving the flags validation are taken from:
;;https://github.com/bhauman/figwheel-main-template/blob/master/src/leiningen/new/figwheel_main.clj
(defn- next-row
  [previous current other-seq]
  (reduce
    (fn [row [diagonal above other]]
      (let [update-val (if (= other current)
                          diagonal
                          (inc (min diagonal above (peek row))))]
        (conj row update-val)))
    [(inc (first previous))]
    (map vector previous (next previous) other-seq)))

(defn- levenshtein
  [sequence1 sequence2]
  (peek
    (reduce (fn [previous current] (next-row previous current sequence2))
            (map #(identity %2) (cons nil sequence2) (range))
            sequence1)))

(defn- similar [ky ky2]
  (let [dist (levenshtein (str ky) (str ky2))]
    (when (<= dist 2) dist)))

(defn similar-options [opt]
  (second (first (sort-by first
                          (filter first (map (juxt (partial similar opt) identity)
                                             flags-opts))))))

(defn parse-opts [opts]
  (reduce (fn [accum opt]
            (cond
              (flags-opts opt) (conj accum (keyword (subs opt 2)))
              :else
              (let [suggestion (similar-options opt)]
                (reduced
                 {:error
                  (format
                   "Unknown option '%s' %s" opt
                   (str
                    (when suggestion
                      (format
                       "\n --> Perhaps you intended to use the '%s' option?" suggestion))))}))))
          #{} opts))

(defn juxt-wedge
  "FIXME: write documentation"
  [name & opts]
  (let [parsed-opts (parse-opts opts)]
    (println "Generating fresh 'clj new' io.dominic.juxt-wedge project.")
    (if (:error parsed-opts)
      (binding [*out* *err*]
        (println (:error parsed-opts))
        (System/exit 1))
      (let [f? parsed-opts
            cljs? (or (f? :cljs) (f? :reframe))
            {:keys [cljs reframe sass kick]
             :as data}
            (assoc (project-data name)

                   :cljs cljs?
                   :reframe (f? :reframe)
                   :sass (f? :sass)
                   :kick (or cljs? (f? :sass))

                   :server-port (generate-port)
                   :figwheel-port (generate-port))
            render (renderer "io.dominic.juxt-wedge")]
        (apply ->files data
               (concat
                 [["deps.edn" (render "deps.edn" data)]
                  ["src/{{nested-dirs}}/core.clj" (render "core.clj" data)]
                  ["src/io/dominic/wedge/system-config.edn"
"{:system io.dominic.wedge.integrant-aero
 :dev {:profile :dev}}"]
                  ["src/config.edn" (render "config.edn" data)]
                  ["src/index.html" (render "index.html" data)]
                  ["dev/dev.clj" (render "dev.clj" data)]
                  ["dev/user.clj" (render "user.clj" data)]
                  ["dev/log_dev_app.properties" (render "log_dev_app.properties" data)]
                  [".clj-kondo/.gitkeep" ""]
                  ;; TODO: Add build to aliases when using kick, also cljs
                  ;; flags
                  [".dir-locals.el" (impl.dev-tooling/dir-locals ["dev"] false)]
                  [".vscode/settings.json" (impl.dev-tooling/calva-settings-json ["dev"] false)]]
                 (when kick [["target/dev/.gitkeep" ""]
                             ["target/prod/.gitkeep" ""]])
                 (if sass
                   [["src/{{name}}.scss" (render "app.css" data)]]
                   [["src/public/{{name}}.css" (render "app.css" data)]])
                 (when cljs
                     [["src/{{nested-dirs}}/frontend/main.cljs"
                       (render (cond
                                 reframe "reframe/main.cljs"
                                 cljs "main.cljs")
                               data)]])
                 (when reframe
                     [["src/{{nested-dirs}}/frontend/views.cljs" (render "reframe/views.cljs" data)]
                      ["src/{{nested-dirs}}/frontend/db.cljs" (render "reframe/db.cljs" data)]
                      ["src/{{nested-dirs}}/frontend/handlers.cljs" (render "reframe/handlers.cljs" data)]
                      ["src/{{nested-dirs}}/frontend/subs.cljs" (render "reframe/subs.cljs" data)]])))))))
