(ns io.dominic.wedge.integrant-aero
  "Wedge-compatible :system for integrant/aero.  Example system_config.edn:
    * {:system io.dominic.wedge.integrant-aero}
    * {:system io.dominic.wedge.integrant-aero
       :key :integrant/system}

  Uses aero to read `config.edn`, and gets the system from it.

  Requires `aero` and `integrant` to be on the classpath, and for dev also
  requires `integrant/repl`.

  Runs `ig/prep` and `ig/load-namespaces` as part of the start process.

  Adds #ig/ref and #ig/refset to Aero, which correspond to ig/ref and ig/refset
  respectively.

  See [[system-config]] for supported options."
  (:require
    [aero.core :as aero]
    [integrant.core :as ig]
    [io.dominic.wedge.impl.aero :as impl.aero]))

(defmethod aero/reader 'ig/ref [_ _ value]
  (ig/ref value))

(defmethod aero/reader 'ig/refset [_ _ value]
  (ig/refset value))

(defn get-config
  "opts will be passed to aero.
   Optional opts:
     :io.dominic.wedge.aero/tag-ns - a list of symbols to be required prior to reading.  Use this to load any namespaces containing Aero tags."
  [opts]
  (impl.aero/config opts))

(defn ^:no-doc prep
  "WARNING: Implementation detail.  Subject to change."
  [config {:keys [key]
           :or {key :ig/system}}]
  (let [system (get config key)]
    (locking clojure.lang.RT/REQUIRE_LOCK
      (ig/load-namespaces system))
    (ig/prep system)
    system))

(defn start
  "Optional opts:
     :key - the key in config which contains the integrant system.  Defaults to
            :ig/system.
     :keys - keys to start."
  [config opts]
  (let [system (prep config opts)]
    (if-let [[_ keys] (find opts :keys)]
      (ig/init system keys)
      (ig/init system))))

(defn stop
  [system]
  (ig/halt! system))

(defn load-dev
  []
  (load "/io/dominic/wedge/impl/integrant_aero_dev"))

(defn load-system
  [config {:keys [key]
           :or {key :ig/system}}]
  (let [system (get config key)]
    (locking clojure.lang.RT/REQUIRE_LOCK
      (ig/load-namespaces system))))
