(ns io.dominic.wedge.user
  "Functions commonly used from a `user` namespace.  Recommended usage is like:
  ```
  (ns user
    (:require [io.dominic.wedge.user :refer :all]))
  ```"
  (:require
    [clojure.edn :as edn]
    [clojure.java.classpath :refer [classpath-directories]]
    [clojure.java.io :as io]
    [clojure.java.shell :as sh]
    [clojure.set :as set]
    [clojure.string :as string]
    [clojure.tools.namespace.repl :as namespace.repl]
    [io.dominic.wedge.dev :as wedge.dev]))

(when (try
        (Class/forName "org.slf4j.bridge.SLF4JBridgeHandler")
        (catch ClassNotFoundException _
          false))
  (eval
    `(do
       (org.slf4j.bridge.SLF4JBridgeHandler/removeHandlersForRootLogger)
       (org.slf4j.bridge.SLF4JBridgeHandler/install))))

(def ^:private basis (some-> (System/getProperty "clojure.basis")
                             io/reader
                             (java.io.PushbackReader.)
                             edn/read))

(doto (Thread.
        (fn []
          (try
            (when (.exists (io/file ".clj-kondo"))
              (sh/sh "clj-kondo" "--no-warnings" "--parallel" "--lint"
                     (string/join java.io.File/pathSeparator (:classpath-roots basis))))
            (catch Throwable _
              ;; Silent error is absolutely fine here.  This is simply a
              ;; convenience for users.
              ,))))
  (.setDaemon true)
  (.setName "wedge-clj-kondo-cacher")
  (.start))

(defonce ^:private original-refresh-dirs
  (or (seq @#'namespace.repl/refresh-dirs) (classpath-directories)))

(let [build-paths (when basis
                    (map (fn [path]
                           (.getCanonicalFile (io/file path)))
                         (mapcat
                           (fn [path]
                             (if (keyword? path)
                               (let [paths (get-in basis [:aliases path])]
                                 (when (sequential? paths)
                                   paths))
                               [path]))
                           (mapcat
                             (fn [[k v]]
                               (when (.contains (str k) ":build")
                                 (cond
                                   (map? v)
                                   (concat (:extra-paths v) (:replace-paths v))
                                   (sequential? v)
                                   v)))
                             (:aliases basis)))))]
  (apply namespace.repl/set-refresh-dirs
         (set/difference
           (set (map #(.getCanonicalFile %) original-refresh-dirs))
           (set build-paths))))

(defn exclude-libs-refresh
  "Exclude paths of lib-names from the tools.namespace refresh-dirs."
  [& lib-names]
  (apply namespace.repl/set-refresh-dirs
         (remove
           (set (map #(.getCanonicalFile (io/file %))
                     (mapcat #(get-in basis [:libs % :paths]) lib-names)))
           @#'namespace.repl/refresh-dirs)))

(defn reset-refresh-dirs
  "Set refresh-dirs to their original value, undoing anything Wedge has done.
  Use this if you'd like to perform your own filtering."
  []
  (apply namespace.repl/set-refresh-dirs original-refresh-dirs))

(let [lock (Object.)]
  (defn dev
    "Call this to switch to the dev namespace."
    []
    (println "[Wedge] Loading dev")
    (locking lock
      (require 'dev))
    (when-not wedge.dev/system
      (println "[Wedge] Enter (go) to start the system"))
    (in-ns 'dev)))

(defn fixed!
  "If, for some reason, the Clojure code in the project fails to
  compile - we still bring up a REPL to help debug the problem. Once
  the problem has been resolved you can call this function to continue
  development."
  []
  (namespace.repl/refresh-all)
  (in-ns 'dev))

(def refresh namespace.repl/refresh)
(def refresh-all namespace.repl/refresh-all)
