(ns io.dominic.wedge.system
  "Functions to operate on Wedge's system infrastructure.
  
  By default, your system is read from a key that's specified by the system.
  e.g. integrant reads from :ig/system.  You may override this by specifying a
  `:key` in your `opts`."
  (:require
    [clojure.java.io :as io]
    [clojure.edn :as edn]))

(defn read-system-config
  "Read system-config.edn, or fallback onto a default system config."
  []
  (if-let [r (io/resource "io/dominic/wedge/system-config.edn")]
    (edn/read (java.io.PushbackReader. (io/reader r)))
    {:system 'io.dominic.wedge.clip
     :config 'io.dominic.wedge.aero
     :dev {:profile :dev}}))

(defn- requiring-resolve*
  "Like requiring-resolve but throws a data error message when namespace or var
  is missing."
  [sym]
  (try (requiring-resolve sym)
       (catch java.io.FileNotFoundException _
         (throw (ex-info (str "Missing " sym)
                         {::nm sym
                          ::namespace (namespace sym)
                          ::type ::missing-ns})))))

(defn- call-configured
  [k nm & args]
  (let [config (read-system-config)
        sym (symbol (name (get config k))
                    (name nm))]
    (if-let [f (requiring-resolve* sym)]
      (apply f args)
      (throw (ex-info (str "Missing " sym)
                      {::nm nm
                       ::config config
                       ::type ::missing-var})))))

(defn- system-k
  [opts]
  (or (:key opts)
      (let [config (read-system-config)
            sym (symbol (name (get config :system)) "key")]
        (or (some-> (requiring-resolve* sym) deref)
            :system))))

(defn- call-system
  [nm & args]
  (apply call-configured :system nm args))

(defn- call-config
  [nm & args]
  (apply call-configured :config nm args))

(defn read-config
  "Return a config, used to influence other functionality such as systems
  or ClojureScript builds."
  [opts]
  (call-config 'read opts))

(defn get-config
  [config k _opts]
  (call-config 'get config k))

(defn system-desc
  "Call [[get-config]] with the k preferred by the system."
  [config opts]
  (get-config config (system-k opts) opts))

(defn start
  "Take system-desc and start it, returning a started system."
  [system-desc opts]
  (call-system 'start system-desc opts))

(defn stop
  "Take started system from [[start]] and stop it."
  [system]
  (call-system 'stop system))

(defn load-dev
  "Load dev functions into `*ns*`"
  []
  (call-system 'load-dev))

(defn load-system
  "Load namespaces needed by system"
  [system opts]
  (call-system 'load-system system opts))
