(ns ^:no-doc io.dominic.wedge.impl.dev-system-hooks
  (:require
    [clojure.stacktrace :as stacktrace]))

(def ^:private fixtures (atom nil))

(defn add-dev-system-fixture!
  "Register function to run with changes to the dev system.  `lifecycle` can be
  one of :start-once, :start-always, :stop-once, :stop-always.  Those ending in
  `-once` will only run if the system is actually changing (i.e. not on reset
  or suspend), `-always` runs on every call to stop/start.  `key` should be a
  globally unique key to identify this fixture.  `f` should be a function of no
  args which will be called when the lifecycle is matched.
  
  Safe to call multiple times, will replace existing functions when doing so."
  [lifecycle key f]
  (swap! fixtures assoc-in [lifecycle key] f)
  :added)

(defn remove-dev-system-fixture!
  "Remove a fixture added with [[add-dev-system-fixture!]]"
  [lifecycle key]
  (swap! fixtures update lifecycle dissoc key)
  :removed)

(def ^:private state (agent false))

(defn- next-state
  [state lifecycle]
  (let [state' (case lifecycle
                 :start-once true
                 :stop-once false
                 state)]
    (when (not= state' state)
      (try
        (run! (fn [[k f]]
                (try
                  (f)
                  (catch Exception e
                    (throw (ex-info
                             (format "Failed to run %s in lifecycle %s.  Skipping remaining lifecycle functions."
                                     (pr-str k)
                                     (pr-str lifecycle))
                             {:k k :lifecycle lifecycle} e)))))
              (get @fixtures lifecycle))
        (catch Throwable t
          (stacktrace/print-cause-trace t))))
    state'))

(defn raise-evt
  [lifecycle]
  (send-off state next-state lifecycle)
  (await state))
