(ns io.dominic.wedge.main
  (:require
    [clojure.main :as main]
    [clojure.edn :as edn]
    [io.dominic.wedge.system :as wedge.system]))

(when (try
        (Class/forName "org.slf4j.bridge.SLF4JBridgeHandler")
        (catch ClassNotFoundException _
          false))
  (eval
    `(do
       (org.slf4j.bridge.SLF4JBridgeHandler/removeHandlersForRootLogger)
       (org.slf4j.bridge.SLF4JBridgeHandler/install))))

(defn- read-args
  [args]
  (mapv
    #(try
       (edn/read-string {:default tagged-literal} %)
       (catch Throwable t
         (throw (ex-info (str "Unreadable arg: " (pr-str %)) {:arg %} t))))
    args))

(defn- parse-args
  [args]
  (cond
    (some '#{--repl -r} args)
    {::repl true}
    (some '#{-h --help} args)
    {::help true}

    :else
    (do
      (cond
        (odd? (count args)) (throw (ex-info (format "Odd number of arguments provided (%d), cannot convert to map" (count args))
                                            {:args args})))
      (reduce
        (fn [acc [k v]]
          (if (sequential? k)
            (assoc-in acc k v)
            (assoc acc k v)))
        {}
        (partition-all 2 args)))))

(def system)

(defn -main
  [& args]
  (let [{::keys [repl help] :as parsed} (parse-args (read-args args))]
    (cond
      repl
      (main/repl)

      help
      (println
"Usage: [opt] [key val]*

With no options or args, runs the wedge system.

opt:
  -h, --help   Show this help
  -r, --repl   Start clojure.main/repl

key val:
  A series of EDN values which will be converted into a map.  If the key is
  sequential, it will be passed to assoc-in instead of assoc.

  The produced map will be passed to wedge.system/system-config when starting
  the wedge system (and ignored otherwise).

  Example:
    clojure -m io.dominic.wedge.main :profile :staging
    clojure -m io.dominic.wedge.main :io.dominic.wedge.integrant/keys [:web-server]
    clojure -m io.dominic.wedge.main [:profile] :staging")

      :else
      (let [system* (wedge.system/start (wedge.system/system-config parsed))]
        (println "System started")
        (alter-var-root #'system (constantly system*))
        (.addShutdownHook
          (Runtime/getRuntime)
          (Thread. #(wedge.system/stop system*)))
        ;; Block forever
        @(promise)))))
