package io.dropwizard.revolver.core.resilience;

import com.codahale.metrics.MetricRegistry;
import com.google.common.collect.Maps;
import io.dropwizard.revolver.core.config.HystrixCommandConfig;
import io.dropwizard.revolver.core.config.RevolverConfig;
import io.dropwizard.revolver.core.config.RevolverServiceConfig;
import io.dropwizard.revolver.core.config.ThreadPoolGroupConfig;
import io.dropwizard.revolver.http.config.RevolverHttpServiceConfig;
import io.github.resilience4j.bulkhead.Bulkhead;
import io.github.resilience4j.bulkhead.BulkheadConfig;
import io.github.resilience4j.bulkhead.BulkheadRegistry;
import io.github.resilience4j.circuitbreaker.CircuitBreakerRegistry;
import io.github.resilience4j.micrometer.tagged.TaggedBulkheadMetrics;
import io.github.resilience4j.micrometer.tagged.TaggedCircuitBreakerMetrics;
import io.micrometer.core.instrument.Clock;
import io.micrometer.core.instrument.dropwizard.DropwizardConfig;
import io.micrometer.core.instrument.dropwizard.DropwizardMeterRegistry;
import io.micrometer.core.instrument.util.HierarchicalNameMapper;
import java.util.Map;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

/***
 Created by nitish.goyal on 23/11/19
 ***/
@Slf4j
public class ResilienceUtil {

    private ResilienceUtil() {
    }

    private static BulkheadRegistry bulkheadRegistry = BulkheadRegistry.ofDefaults();
    private static CircuitBreakerRegistry circuitBreakerRegistry =
            CircuitBreakerRegistry.ofDefaults();

    public static void initializeResilience(RevolverConfig revolverConfig,
            ResilienceHttpContext resilienceHttpContext, MetricRegistry metrics) {

        log.info("Initializing resilience util");
        io.micrometer.core.instrument.MeterRegistry metricRegistry = new DropwizardMeterRegistry(
                new DropwizardConfig() {
                    @Override
                    public String prefix() {
                        return "resilience";
                    }

                    @Override
                    public String get(String s) {
                        return s;
                    }
                }, metrics,
                HierarchicalNameMapper.DEFAULT, Clock.SYSTEM) {
            @Override
            protected Double nullGaugeValue() {
                return null;
            }
        };

        TaggedBulkheadMetrics
                .ofBulkheadRegistry(bulkheadRegistry)
                .bindTo(metricRegistry);

        TaggedCircuitBreakerMetrics
                .ofCircuitBreakerRegistry(circuitBreakerRegistry)
                .bindTo(metricRegistry);

        initializeBulkHeads(revolverConfig, resilienceHttpContext);
        initializeCircuitBreakers(resilienceHttpContext);
        initializeTimeout(revolverConfig, resilienceHttpContext);
    }

    private static void initializeCircuitBreakers(
            ResilienceHttpContext resilienceHttpContext) {
        resilienceHttpContext.setCircuitBreaker(circuitBreakerRegistry.circuitBreaker("revolver"));
    }

    private static void initializeBulkHeads(RevolverConfig revolverConfig,
            ResilienceHttpContext resilienceHttpContext) {
        Map<String, Bulkhead> poolVsBulkHead = Maps.newHashMap();

        for (RevolverServiceConfig revolverServiceConfig : revolverConfig.getServices()) {

            ThreadPoolGroupConfig threadPoolGroupConfig = revolverServiceConfig.getThreadPoolGroupConfig();
            if (threadPoolGroupConfig != null) {
                threadPoolGroupConfig.getThreadPools().forEach(threadPoolConfig -> {
                    poolVsBulkHead.put(threadPoolConfig.getThreadPoolName(),
                            bulkheadRegistry.bulkhead(threadPoolConfig.getThreadPoolName(),
                                    BulkheadConfig.custom().maxConcurrentCalls(threadPoolConfig.getConcurrency())
                                            .build()));
                });
            }

            if (revolverServiceConfig instanceof RevolverHttpServiceConfig) {
                ((RevolverHttpServiceConfig) revolverServiceConfig).getApis()
                        .forEach(revolverHttpApiConfig -> {
                            if (revolverHttpApiConfig.getRuntime() != null) {
                                HystrixCommandConfig hystrixCommandConfig = revolverHttpApiConfig.getRuntime();
                                if (hystrixCommandConfig == null || hystrixCommandConfig.getThreadPool() == null) {
                                    return;
                                }
                                String threadPoolName = hystrixCommandConfig.getThreadPool().getThreadPoolName();
                                if (StringUtils.isEmpty(threadPoolName)) {
                                    threadPoolName =
                                            revolverServiceConfig.getService() + "." + revolverHttpApiConfig.getApi();
                                }
                                poolVsBulkHead.putIfAbsent(threadPoolName,
                                        bulkheadRegistry.bulkhead(
                                                threadPoolName,
                                                BulkheadConfig.custom().maxConcurrentCalls(
                                                        hystrixCommandConfig.getThreadPool()
                                                                .getConcurrency())
                                                        .build()));
                            }
                        });
            }
        }

        resilienceHttpContext.setPoolVsBulkHeadMap(poolVsBulkHead);
    }

    private static void initializeTimeout(RevolverConfig revolverConfig,
            ResilienceHttpContext resilienceHttpContext) {
        Map<String, Integer> poolVsTimeout = Maps.newHashMap();

        for (RevolverServiceConfig revolverServiceConfig : revolverConfig.getServices()) {

            ThreadPoolGroupConfig threadPoolGroupConfig = revolverServiceConfig.getThreadPoolGroupConfig();
            if (threadPoolGroupConfig != null) {
                threadPoolGroupConfig.getThreadPools().forEach(threadPoolConfig -> {
                    poolVsTimeout.put(threadPoolConfig.getThreadPoolName(),
                            threadPoolConfig.getTimeout());
                });
            }

            if (revolverServiceConfig instanceof RevolverHttpServiceConfig) {
                ((RevolverHttpServiceConfig) revolverServiceConfig).getApis()
                        .forEach(revolverHttpApiConfig -> {
                            if (revolverHttpApiConfig.getRuntime() != null) {
                                HystrixCommandConfig hystrixCommandConfig = revolverHttpApiConfig.getRuntime();
                                if (hystrixCommandConfig == null || hystrixCommandConfig.getThreadPool() == null) {
                                    return;
                                }
                                String threadPoolName = hystrixCommandConfig.getThreadPool().getThreadPoolName();
                                if (StringUtils.isEmpty(threadPoolName)) {
                                    threadPoolName =
                                            revolverServiceConfig.getService() + "." + revolverHttpApiConfig.getApi();
                                }
                                poolVsTimeout.putIfAbsent(threadPoolName,
                                        hystrixCommandConfig.getThreadPool().getTimeout());
                            }
                        });
            }
        }

        resilienceHttpContext.setPoolVsTimeout(poolVsTimeout);
    }

}
