(ns himmelsstuermer.core.db
  (:require
    [malli.core :as malli]))


(defprotocol DBBackendProtocol

  (get-store-cfg [this])

  (run-query [this query db args])

  (is-database? [this obj])

  (is-connection? [this obj])

  (database-exists? [this])

  (create-database [this schema])

  (get-connection [this schema])

  (transaction [this cn tx-data]))


(defrecord DBBackendDummy
  []

  DBBackendProtocol

  (get-store-cfg
    [_]
    (throw (Exception. "No database backend!")))


  (run-query
    [_ _ _ _]
    (throw (Exception. "No database backend!")))


  (is-database?
    [_ _]
    (throw (Exception. "No database backend!")))


  (is-connection?
    [_ _]
    (throw (Exception. "No database backend!")))


  (database-exists?
    [_]
    (throw (Exception. "No database backend!")))


  (create-database
    [_ _]
    (throw (Exception. "No database backend!")))


  (get-connection
    [_ _]
    (throw (Exception. "No database backend!")))


  (transaction
    [_ _ _]
    (throw (Exception. "No database backend!"))))


(defonce ^:private db-backend (atom (->DBBackendDummy)))


(malli/=> set-database-backend! [:-> [:fn #(satisfies? DBBackendProtocol %)] :nil])


(defn set-database-backend!
  [backend]
  (println "\nDatabase backend set: " (type backend) "\n")
  (reset! db-backend backend))


(defn get-config
  []
  (get-store-cfg @db-backend))


(defn q
  [query db & args]
  (run-query @db-backend query db args))


(defn is-db?
  [obj]
  (is-database? @db-backend obj))


(defn is-conn?
  [obj]
  (is-connection? @db-backend obj))


(defn db-exists?
  []
  (database-exists? @db-backend))


(defn create-db
  [schema]
  (create-database @db-backend schema))


(defn connect
  [schema]
  (get-connection @db-backend schema))


(defn transact
  [state tx-set]
  (transaction @db-backend (-> state :system :db-conn) (vec tx-set)))
