(ns conjurernix.malli-jpa.api
  (:require [conjurernix.malli-pojo.api :as mp]
            [malli.core :as m])
  (:import (javax.persistence Column Entity EnumType Enumerated
                              GeneratedValue GenerationType Id
                              Table Temporal TemporalType Transient)))

(def ->generation-type
  {:auto     GenerationType/AUTO
   :identity GenerationType/IDENTITY
   :sequence GenerationType/SEQUENCE
   :table    GenerationType/TABLE})

(def ->temporal-type
  {:date      TemporalType/DATE
   :time      TemporalType/TIME
   :timestamp TemporalType/TIMESTAMP})

(def ->enum-type
  {:string  EnumType/STRING
   :ordinal EnumType/ORDINAL})

(defn entity
  "Given a malli schema, retrieve the associated Entity class."
  [s]
  (mp/schema->pojo s))

(defmacro defentity [& [name props schema :as args]]
  (let [{::keys [entity-name table class-name]} props
        {table-name   :name
         table-schema :schema} table
        class-annotations (cond-> {Entity (cond-> {}
                                            entity-name (assoc :name entity-name))}
                            table-name (assoc-in [Table :name] table-name)
                            table-schema (assoc-in [Table :schema] table-schema))

        getter-annotations (->> schema
                                (m/children)
                                (map (fn [[k {::keys [id generated-value column temporal enumerated transient]} _v]]
                                       [k (cond-> {}

                                            id (assoc Id {})

                                            generated-value
                                            (assoc GeneratedValue
                                                   ; TODO: all Column properties must be handled
                                                   (let [{generated-value-strategy :strategy} generated-value]
                                                     (cond-> {}
                                                       generated-value-strategy
                                                       (assoc :strategy (->generation-type generated-value-strategy)))))

                                            column
                                            (assoc Column
                                                   ; TODO: some keys like length nullable can be infered from malli schema
                                                   ; TODO: all Column properties must be handled
                                                   (let [{:keys [name length nullable unique]} column]
                                                     (cond-> {}
                                                       name (assoc :name name)
                                                       length (assoc :length length)
                                                       nullable (assoc :nullable nullable)
                                                       unique (assoc :unique unique))))

                                            temporal
                                            (assoc Temporal (->temporal-type temporal))

                                            transient
                                            (assoc Transient true)

                                            enumerated
                                            (assoc Enumerated (->enum-type enumerated))

                                            )]))
                                (into {}))

        pojo-props
        {:class-name  class-name
         :annotations {:class  class-annotations
                       :getter getter-annotations}}]
    `(do
       (mp/defpojo ~name ~pojo-props ~schema))))
