(ns dots.node.fs.stat-watcher
  "Class: fs.StatWatcher")

(defn ref
  "When called, requests that the Node.js event loop _not_ exit so long as the `fs.StatWatcher` is active. Calling `watcher.ref()` multiple times will have
   no effect.
   
   By default, all `fs.StatWatcher` objects are \"ref'ed\", making it normally
   unnecessary to call `watcher.ref()` unless `watcher.unref()` had been
   called previously.
   
   **Returns:** `StatWatcher`"
  ^js [stat-watcher]
  (.ref ^js stat-watcher))

(defn unref
  "When called, the active `fs.StatWatcher` object will not require the Node.js
   event loop to remain active. If there is no other activity keeping the
   event loop running, the process may exit before the `fs.StatWatcher` object's
   callback is invoked. Calling `watcher.unref()` multiple times will have
   no effect.
   
   **Returns:** `StatWatcher`"
  ^js [stat-watcher]
  (.unref ^js stat-watcher))

(defn add-listener
  "Alias for `emitter.on(eventName, listener)`.
   
   **Parameters:**
   - `event-name`: `string | symbol`
   - `listener`: `(...args: any[]) => void`
   
   **Returns:** `StatWatcher`"
  ^js [stat-watcher event-name listener]
  (.addListener ^js stat-watcher event-name listener))

(defn on
  "Adds the `listener` function to the end of the listeners array for the event
   named `eventName`. No checks are made to see if the `listener` has already
   been added. Multiple calls passing the same combination of `eventName` and
   `listener` will result in the `listener` being added, and called, multiple times.
   
   ```js
   server.on('connection', (stream) => {
     console.log('someone connected!');
   });
   ```
   
   Returns a reference to the `EventEmitter`, so that calls can be chained.
   
   By default, event listeners are invoked in the order they are added. The `emitter.prependListener()` method can be used as an alternative to add the
   event listener to the beginning of the listeners array.
   
   ```js
   import { EventEmitter } from 'node:events';
   const myEE = new EventEmitter();
   myEE.on('foo', () => console.log('a'));
   myEE.prependListener('foo', () => console.log('b'));
   myEE.emit('foo');
   // Prints:
   //   b
   //   a
   ```
   
   **Parameters:**
   - `event-name`: `string | symbol` - The name of the event.
   - `listener`: `(...args: any[]) => void` - The callback function
   
   **Returns:** `StatWatcher`"
  ^js [stat-watcher event-name listener]
  (.on ^js stat-watcher event-name listener))

(defn once
  "Adds a **one-time** `listener` function for the event named `eventName`. The
   next time `eventName` is triggered, this listener is removed and then invoked.
   
   ```js
   server.once('connection', (stream) => {
     console.log('Ah, we have our first user!');
   });
   ```
   
   Returns a reference to the `EventEmitter`, so that calls can be chained.
   
   By default, event listeners are invoked in the order they are added. The `emitter.prependOnceListener()` method can be used as an alternative to add the
   event listener to the beginning of the listeners array.
   
   ```js
   import { EventEmitter } from 'node:events';
   const myEE = new EventEmitter();
   myEE.once('foo', () => console.log('a'));
   myEE.prependOnceListener('foo', () => console.log('b'));
   myEE.emit('foo');
   // Prints:
   //   b
   //   a
   ```
   
   **Parameters:**
   - `event-name`: `string | symbol` - The name of the event.
   - `listener`: `(...args: any[]) => void` - The callback function
   
   **Returns:** `StatWatcher`"
  ^js [stat-watcher event-name listener]
  (.once ^js stat-watcher event-name listener))

(defn remove-listener
  "Removes the specified `listener` from the listener array for the event named `eventName`.
   
   ```js
   const callback = (stream) => {
     console.log('someone connected!');
   };
   server.on('connection', callback);
   // ...
   server.removeListener('connection', callback);
   ```
   
   `removeListener()` will remove, at most, one instance of a listener from the
   listener array. If any single listener has been added multiple times to the
   listener array for the specified `eventName`, then `removeListener()` must be
   called multiple times to remove each instance.
   
   Once an event is emitted, all listeners attached to it at the
   time of emitting are called in order. This implies that any `removeListener()` or `removeAllListeners()` calls _after_ emitting and _before_ the last listener finishes execution
   will not remove them from`emit()` in progress. Subsequent events behave as expected.
   
   ```js
   import { EventEmitter } from 'node:events';
   class MyEmitter extends EventEmitter {}
   const myEmitter = new MyEmitter();
   
   const callbackA = () => {
     console.log('A');
     myEmitter.removeListener('event', callbackB);
   };
   
   const callbackB = () => {
     console.log('B');
   };
   
   myEmitter.on('event', callbackA);
   
   myEmitter.on('event', callbackB);
   
   // callbackA removes listener callbackB but it will still be called.
   // Internal listener array at time of emit [callbackA, callbackB]
   myEmitter.emit('event');
   // Prints:
   //   A
   //   B
   
   // callbackB is now removed.
   // Internal listener array [callbackA]
   myEmitter.emit('event');
   // Prints:
   //   A
   ```
   
   Because listeners are managed using an internal array, calling this will
   change the position indices of any listener registered _after_ the listener
   being removed. This will not impact the order in which listeners are called,
   but it means that any copies of the listener array as returned by
   the `emitter.listeners()` method will need to be recreated.
   
   When a single function has been added as a handler multiple times for a single
   event (as in the example below), `removeListener()` will remove the most
   recently added instance. In the example the `once('ping')` listener is removed:
   
   ```js
   import { EventEmitter } from 'node:events';
   const ee = new EventEmitter();
   
   function pong() {
     console.log('pong');
   }
   
   ee.on('ping', pong);
   ee.once('ping', pong);
   ee.removeListener('ping', pong);
   
   ee.emit('ping');
   ee.emit('ping');
   ```
   
   Returns a reference to the `EventEmitter`, so that calls can be chained.
   
   **Parameters:**
   - `event-name`: `string | symbol`
   - `listener`: `(...args: any[]) => void`
   
   **Returns:** `StatWatcher`"
  ^js [stat-watcher event-name listener]
  (.removeListener ^js stat-watcher event-name listener))

(defn off
  "Alias for `emitter.removeListener()`.
   
   **Parameters:**
   - `event-name`: `string | symbol`
   - `listener`: `(...args: any[]) => void`
   
   **Returns:** `StatWatcher`"
  ^js [stat-watcher event-name listener]
  (.off ^js stat-watcher event-name listener))

(defn remove-all-listeners
  "Removes all listeners, or those of the specified `eventName`.
   
   It is bad practice to remove listeners added elsewhere in the code,
   particularly when the `EventEmitter` instance was created by some other
   component or module (e.g. sockets or file streams).
   
   Returns a reference to the `EventEmitter`, so that calls can be chained.
   
   **Parameters:**
   - `event-name`: `string | symbol | undefined`
   
   **Returns:** `StatWatcher`"
  (^js [stat-watcher]
   (.removeAllListeners ^js stat-watcher))
  (^js [stat-watcher event-name]
   (.removeAllListeners ^js stat-watcher event-name)))

(defn set-max-listeners
  "By default `EventEmitter`s will print a warning if more than `10` listeners are
   added for a particular event. This is a useful default that helps finding
   memory leaks. The `emitter.setMaxListeners()` method allows the limit to be
   modified for this specific `EventEmitter` instance. The value can be set to `Infinity` (or `0`) to indicate an unlimited number of listeners.
   
   Returns a reference to the `EventEmitter`, so that calls can be chained.
   
   **Parameters:**
   - `n`: `number`
   
   **Returns:** `StatWatcher`"
  ^js [stat-watcher n]
  (.setMaxListeners ^js stat-watcher n))

(defn max-listeners
  "Returns the current max listener value for the `EventEmitter` which is either
   set by `emitter.setMaxListeners(n)` or defaults to {@link defaultMaxListeners }.
   
   **Returns:** `number`"
  ^js [stat-watcher]
  (.getMaxListeners ^js stat-watcher))

(defn listeners
  "Returns a copy of the array of listeners for the event named `eventName`.
   
   ```js
   server.on('connection', (stream) => {
     console.log('someone connected!');
   });
   console.log(util.inspect(server.listeners('connection')));
   // Prints: [ [Function] ]
   ```
   
   **Parameters:**
   - `event-name`: `string | symbol`
   
   **Returns:** `Function[]`"
  ^js [stat-watcher event-name]
  (.listeners ^js stat-watcher event-name))

(defn raw-listeners
  "Returns a copy of the array of listeners for the event named `eventName`,
   including any wrappers (such as those created by `.once()`).
   
   ```js
   import { EventEmitter } from 'node:events';
   const emitter = new EventEmitter();
   emitter.once('log', () => console.log('log once'));
   
   // Returns a new Array with a function `onceWrapper` which has a property
   // `listener` which contains the original listener bound above
   const listeners = emitter.rawListeners('log');
   const logFnWrapper = listeners[0];
   
   // Logs \"log once\" to the console and does not unbind the `once` event
   logFnWrapper.listener();
   
   // Logs \"log once\" to the console and removes the listener
   logFnWrapper();
   
   emitter.on('log', () => console.log('log persistently'));
   // Will return a new Array with a single function bound by `.on()` above
   const newListeners = emitter.rawListeners('log');
   
   // Logs \"log persistently\" twice
   newListeners[0]();
   emitter.emit('log');
   ```
   
   **Parameters:**
   - `event-name`: `string | symbol`
   
   **Returns:** `Function[]`"
  ^js [stat-watcher event-name]
  (.rawListeners ^js stat-watcher event-name))

(defn emit?
  "Synchronously calls each of the listeners registered for the event named `eventName`, in the order they were registered, passing the supplied arguments
   to each.
   
   Returns `true` if the event had listeners, `false` otherwise.
   
   ```js
   import { EventEmitter } from 'node:events';
   const myEmitter = new EventEmitter();
   
   // First listener
   myEmitter.on('event', function firstListener() {
     console.log('Helloooo! first listener');
   });
   // Second listener
   myEmitter.on('event', function secondListener(arg1, arg2) {
     console.log(`event with parameters ${arg1}, ${arg2} in second listener`);
   });
   // Third listener
   myEmitter.on('event', function thirdListener(...args) {
     const parameters = args.join(', ');
     console.log(`event with parameters ${parameters} in third listener`);
   });
   
   console.log(myEmitter.listeners('event'));
   
   myEmitter.emit('event', 1, 2, 3, 4, 5);
   
   // Prints:
   // [
   //   [Function: firstListener],
   //   [Function: secondListener],
   //   [Function: thirdListener]
   // ]
   // Helloooo! first listener
   // event with parameters 1, 2 in second listener
   // event with parameters 1, 2, 3, 4, 5 in third listener
   ```
   
   **Parameters:**
   - `event-name`: `string | symbol`
   - `args`: `AnyRest`
   
   **Returns:** `boolean`"
  ^js [stat-watcher event-name & args]
  (.. ^js stat-watcher -emit (apply ^js stat-watcher (to-array (cons event-name args)))))

(defn listener-count
  "Returns the number of listeners listening for the event named `eventName`.
   If `listener` is provided, it will return how many times the listener is found
   in the list of the listeners of the event.
   
   **Parameters:**
   - `event-name`: `string | symbol` - The name of the event being listened for
   - `listener`: `Function | undefined` - The event handler function
   
   **Returns:** `number`"
  (^js [stat-watcher event-name]
   (.listenerCount ^js stat-watcher event-name))
  (^js [stat-watcher event-name listener]
   (.listenerCount ^js stat-watcher event-name listener)))

(defn prepend-listener
  "Adds the `listener` function to the _beginning_ of the listeners array for the
   event named `eventName`. No checks are made to see if the `listener` has
   already been added. Multiple calls passing the same combination of `eventName`
   and `listener` will result in the `listener` being added, and called, multiple times.
   
   ```js
   server.prependListener('connection', (stream) => {
     console.log('someone connected!');
   });
   ```
   
   Returns a reference to the `EventEmitter`, so that calls can be chained.
   
   **Parameters:**
   - `event-name`: `string | symbol` - The name of the event.
   - `listener`: `(...args: any[]) => void` - The callback function
   
   **Returns:** `StatWatcher`"
  ^js [stat-watcher event-name listener]
  (.prependListener ^js stat-watcher event-name listener))

(defn prepend-once-listener
  "Adds a **one-time**`listener` function for the event named `eventName` to the _beginning_ of the listeners array. The next time `eventName` is triggered, this
   listener is removed, and then invoked.
   
   ```js
   server.prependOnceListener('connection', (stream) => {
     console.log('Ah, we have our first user!');
   });
   ```
   
   Returns a reference to the `EventEmitter`, so that calls can be chained.
   
   **Parameters:**
   - `event-name`: `string | symbol` - The name of the event.
   - `listener`: `(...args: any[]) => void` - The callback function
   
   **Returns:** `StatWatcher`"
  ^js [stat-watcher event-name listener]
  (.prependOnceListener ^js stat-watcher event-name listener))

(defn event-names
  "Returns an array listing the events for which the emitter has registered
   listeners. The values in the array are strings or `Symbol`s.
   
   ```js
   import { EventEmitter } from 'node:events';
   
   const myEE = new EventEmitter();
   myEE.on('foo', () => {});
   myEE.on('bar', () => {});
   
   const sym = Symbol('symbol');
   myEE.on(sym, () => {});
   
   console.log(myEE.eventNames());
   // Prints: [ 'foo', 'bar', Symbol(symbol) ]
   ```
   
   **Returns:** `(string | symbol)[]`"
  ^js [stat-watcher]
  (.eventNames ^js stat-watcher))
