(ns missinterpret.flows.workflow
  (:require [clojure.pprint :refer [pprint]]
            [manifold.stream :as s]
            [missinterpret.anomalies.anomaly :refer [throw+ anomaly anomaly? wrap-exception]]
            [missinterpret.flows.globals :as globals]
            [missinterpret.flows.core :as core]
            [missinterpret.flows.predicates :as p]
            [missinterpret.flows.utils :as utils.flows]
            [missinterpret.flows.predicates :refer [workflow? workflow-loaded?]]
            [missinterpret.flows.constructors :as cstr]
            [missinterpret.flows.schema :as schema.flows]
            [missinterpret.flows.spec :as spec]))

(defn validate-schema
  "Validates the 'linearized' workflow's schema integrity against the
   schema arguments.

   returns:
    Invalid:  [{integrity map}]
    Integrity data otherwise"
  [integrity {:workflow/keys [schema]}]
  (let [{:keys [required minimum-rank partial-invalid]} schema
        rank (:rank integrity)
        score (schema.flows/score rank)
        minimum-score (schema.flows/score minimum-rank)]
    (cond
      (and (true? required)
           (schema.flows/unspecified? (:graph integrity)))
      [integrity]

      (and (true? partial-invalid)
           (schema.flows/underspecified? integrity))
      [integrity]

      (and (some? minimum-rank)
           (< score minimum-score))
      [integrity]

      :else integrity)))


(defn workflow-factory
  "Converts a workflow definition into an entity which is ready for
   stream operations by using the returned :workflow/source and :workflow/sink
   arguments.

   Topology

   Determines the order of flow execution of the definition.
   The topology is linear unless another ordering is imposed.

   Options
    - :id           Adds the id to the workflow
    - :order        Sets the topology of the execution order

     :linear       (default) The topology is in the order provided by the definition vector
     :interceptor  The call stack will match the interceptor-chain calling order
                    when :flow/before-fn and/or :flow/after-fn exist in any of the
                    definitions flows.

    - :expand       Passes all the results from a sequential value into the downstream
                    flow individually."
  ([workflow]
   (workflow-factory workflow {}))
  ([workflow flow-args]
   (workflow-factory {} workflow flow-args))
  ([flow-catalog
   {:workflow/keys [definition id schema expand order log-fn]
    :or {id nil expand false order :linear} :as arg-wf}
   {:flows/keys [source sink] :as factory-args}]
  (let [topo-validation (core/validate-topology flow-catalog definition)]
    (if (:valid topo-validation)
      (let [linear (cond
                     (= order :linear)      (core/linear-order flow-catalog definition :expand expand)
                     (= order :interceptor) (core/interceptor-order flow-catalog definition :expand expand))
            {:keys [src snk]}
            (cond
              (and (s/stream? source) (s/stream? sink)) {:src source :snk sink}
              (s/stream? source)                        {:src source :snk source}
              (s/stream? sink)                          {:src sink :snk sink}
              :else
              (let [snk-src (s/stream globals/buffer-size)] {:src snk-src :snk snk-src}))
            load-wf (cstr/workflow id linear :workflow/sink src :workflow/source snk)
            integrity (-> (schema.flows/integrity load-wf)
                          (validate-schema schema))]
        (if (sequential? integrity)
          #:workflow{:invalid integrity}
          (let [loaded (loop [nodes linear
                              wf load-wf]
                         (let [n (first nodes)]
                           (if (nil? n)
                             wf
                             (let [loaded (core/connect-node wf n factory-args)]
                               (if (contains? loaded :invalid)
                                 loaded
                                 (recur (rest nodes) loaded))))))]
            (with-meta loaded integrity))))
      #:workflow{:invalid (:invalid topo-validation)}))))


(defn load
  "Loads and connects the flows in the workflow definition adding
   the manifold source/sink making it compatible with run unless
   either `defer-load` and `lazy-load` are true.

   Note: The `force` option will always load the workflow definition."
  ([workflow]
   (load {} workflow))
  ([flow-catalog {:workflow/keys [defer-load lazy-load args force log-fn] :as rt-args}]
   (utils.flows/safe-log log-fn {:from ::load :message {:flow-catalog flow-catalog :workflow rt-args}})
   (cond
     (not (-> rt-args (utils.flows/extract spec/Workflow) workflow?))
     (throw+
       {:from     ::load
        :category :anomaly.category/invalid
        :message  {:readable "Workflow is invalid"
                   :reasons  [:invalid/workflow]
                   :data     {:flow-catalog flow-catalog
                              :runtime-args rt-args}}})

     (or (true? force) (not (or (true? defer-load) (true? lazy-load))))
     (let [flow-args (if (map? args) args {})
           wf-rt (-> (utils.flows/extract rt-args spec/Workflow)
                     (dissoc :workflow/source)
                     (dissoc :workflow/sink))
           loaded-wf (workflow-factory flow-catalog wf-rt flow-args)
           loaded (merge loaded-wf wf-rt)]
       (utils.flows/safe-log log-fn {:from ::load :message {:loaded-wf loaded-wf :loaded loaded}})
       (if (p/workflow-loaded? loaded)
         loaded
         (throw+
           {:from     ::load
            :category :anomaly.category/invalid
            :message  {:readable "Workflow is invalid"
                       :reasons  [:invalid/workflow]
                       :data     {:flow-catalog flow-catalog
                                  :runtime-args rt-args
                                  :flow-args flow-args
                                  :loaded loaded}}})))

     :else (utils.flows/extract rt-args spec/Workflow))))


(defn- run-throw-or-return
  [return-error message]
  (let [anom {:from     ::run
              :category :anomaly.category/invalid
              :message  message}]
    (if (true? return-error)
      (anomaly anom)
      (throw+ anom))))

(defn run
  "Executes a loaded workflow returning the results.

   Options:

   - expand       If the runtime argument is a seq the contents of the seq are
                  individually added.
   - return-n     Returns exactly n values from the workflow results or fails if unable to do so.
   - return-nth   Returns the nth value of the result if it is a seq, fails if unable to do so.

   - repeat-n     Injects n copies of the runtime argument.
   - put-timeout  Sets the amount of time `run` will wait before considering the operation failed.
   - take-timeout Sets the amount of time `run` will wait before considering the operation failed.
   - return-error Returns an anomaly instead of throwing an exception."
  ([workflow]
   (run workflow {}))
  ([{:workflow/keys [expand
                     take-range
                     return-vector
                     return-error
                     schema
                     log-fn]
     :or {expand false take-range nil} :as workflow}
    runtime-arg]
   (utils.flows/safe-log log-fn {:from ::run :message {:workflow workflow :runtime-arg runtime-arg}})
   (let [error-fn (partial run-throw-or-return return-error)]

     ;; Runtime args required by the schema only apply to maps
     (when (and (map? runtime-arg) (-> schema :enforce-runtime-keys true?))
       (let [rt-keys (some->> (keys runtime-arg) (into #{}))
             missing (reduce
                       (fn [coll k]
                         (if (contains? rt-keys k)
                           coll
                           (conj coll k)))
                       #{}
                       (-> (schema.flows/extract-integrity workflow) :trace :runtime keys))]
         (utils.flows/safe-log log-fn {:from ::run :message {:rt-keys rt-keys :missing missing :schema schema}})
         (when (seq missing)
           (error-fn {:readable "Runtime keys expected by schema integrity are missing"
                      :reasons [:invalid/arguments]
                      :data {:workflow workflow :runtime-arg runtime-arg :missing missing}}))))

     (cond
       (not (workflow? workflow))
       (error-fn {:readable "Workflow is invalid"
                  :reasons [:invalid/workflow]
                  :data {:workflow workflow}})

       (not (workflow-loaded? workflow))
       (error-fn {:readable "Workflow is not loaded"
                  :reasons [:invalid.workflow/unloaded]
                  :data {:workflow workflow}})

       ;; Expand; check arg is sequential
       (and (true? expand) (not (sequential? runtime-arg)))
       (error-fn {:readable "Expand can only be used with a sequential runtime argument"
                  :reasons  [:invalid.run/expand]
                  :data     {:workflow workflow :runtime-arg runtime-arg}})

       :else
       (try
         (let [results (core/invoke workflow runtime-arg)
               results-count (count results)]
           (utils.flows/safe-log log-fn {:from ::run :message {:results results}})
           (cond
             (anomaly? results) results

             (and (seq results) (some anomaly? results))
             (let [anoms (filter #(anomaly? %) results)]
               (if (= 1 (count anoms))
                 (if (true? return-error)
                   (first anoms)
                   (throw+ (first anoms)))
                 (error-fn {:readable "Run of workflow contained anomalies in the results"
                            :reasons  [:invalid/run]
                            :data     {:workflow workflow :results results}})))

             :else
             (cond
               (and
                 (some? take-range)
                 (or (and
                       (integer? take-range)
                       (neg? take-range))
                     (and
                       (vector? take-range)
                       (neg? (first take-range))
                       (> (last take-range) results-count))))
               (error-fn {:readable "boundaries of take-range don't match size of results"
                          :reasons  [:invalid/take-range]
                          :data     {:workflow workflow :results results}})

               (some? take-range)    (utils.flows/take-range results take-range)
               (true? return-vector) results

               :else
               (if (> results-count 1)
                 (vec results)
                 (first results)))))

         (catch Exception e (do
                              (utils.flows/safe-log log-fn {:from ::run :message {:execption e}})
                              (if (true? return-error)
                                (wrap-exception e ::run)
                                (throw e)))))))))

