(ns missinterpret.flows.predicates
  (:require [manifold.stream :as s]
            [malli.core :as m]
            [missinterpret.flows.spec :as spec]
            [missinterpret.flows.utils :refer [extract]]))

;; Flow ----------------------------------------------------

(defn default-fn? [fn-arg]
  (when (fn? fn-arg)
    (map? (fn-arg))))

(defn flow? [arg]
  (m/validate spec/Flow arg))

(defn flow= [f1 f2]
  (= (-> (extract f1 spec/Flow) (dissoc :flow/uuid))
     (-> (extract f2 spec/Flow) (dissoc :flow/uuid))))

(defn flow-equiv?
  "Do these two flows have the same flow/fn and optional arguments?"
  [f1 f2]
  (let [flow1 (-> (extract f1 spec/Flow)
                  (dissoc :flow/id)
                  (dissoc :flow/uuid))
        flow2 (-> (extract f2 spec/Flow)
                  (dissoc :flow/id)
                  (dissoc :flow/uuid))]
    (= flow1 flow2)))

(defn flow-fn?
  "Does this flow's fn return a stream when passed the arguments?"
  [{:keys [:flow/fn] :as flow} args]
  (try
    (let [flow-fn (:flow/fn flow)
          source (flow-fn args (s/stream))]
      (and (some? source) (s/stream? source) (s/sink? source)))
    (catch Exception _ false)))


;; Workflow -----------------------------------------------

(defn logging-fn? [lfn]
  (try
    (lfn {})
    true
    (catch Exception _ false)))

(defn factory-result? [wf]
  (or (m/validate spec/Workflow-Factory-Success wf)
      (m/validate spec/Workflow-Factory-Failed wf)))

(defn workflow-loaded? [wf]
  (m/validate spec/Workflow-Factory-Success wf))

(defn workflow-unloaded? [wf]
  (not (workflow-loaded? wf)))

(defn workflow-factory-failed? [wf]
  (m/validate spec/Workflow-Factory-Failed wf))


(defn workflow? [wf]
  (m/validate spec/Workflow wf))

(defn workflow= [wf1 wf2]
  (= (-> (extract wf1 spec/Workflow) (dissoc :workflow/uuid))
     (-> (extract wf2 spec/Workflow) (dissoc :workflow/uuid))))


(defn wf-definition-equiv?
  "Do these two workflows have the same definition?

  Checks:
   - Count matches?
   - Each element in order between the two are:
      1. If keyword =
      2. If flow, flow-equiv?
      3. If workflow, wf-definition-equiv?"
  [wf1 wf2]
  (let [def1 (:workflow/definition wf1)
        def2 (:workflow/definition wf2)]
    (and (= (count def1) (count def2))
         (loop [d1 def1
                d2 def2]
           (let [f1 (first d1)
                 f2 (first d2)]
             (cond
               (or (nil? f1) (nil? f2)) true

               (and (keyword? f1) (keyword? f2) (= f1 f2))
               (recur (rest d1) (rest d2))

               (and (flow? f1) (flow? f2) (flow-equiv? f1 f2))
               (recur (rest d1) (rest d2))

               (and (workflow? f1) (workflow? f2) (wf-definition-equiv? f1 f2))
               (recur (rest d1) (rest d2))

               :else false))))))

(defn workflow-equiv?
  "Do these two workflow have the same definition and optional arguments?

  Excludes: source and sink if present"
  [wf1 wf2]
  (let [workf1 (-> (extract wf1 spec/Workflow)
                   (dissoc :workflow/id)
                   (dissoc :workflow/uuid)
                   (dissoc :workflow/definition)
                   (dissoc :workflow/defer-load)
                   (dissoc :workflow/lazy-load)
                   (dissoc :workflow/new)
                   (dissoc :workflow/force)
                   (dissoc :workflow/source)
                   (dissoc :workflow/sink))
        workf2 (-> (extract wf2 spec/Workflow)
                   (dissoc :workflow/id)
                   (dissoc :workflow/uuid)
                   (dissoc :workflow/definition)
                   (dissoc :workflow/defer-load)
                   (dissoc :workflow/lazy-load)
                   (dissoc :workflow/new)
                   (dissoc :workflow/force)
                   (dissoc :workflow/source)
                   (dissoc :workflow/sink))]
    (and (wf-definition-equiv? wf1 wf2)
         (= workf1 workf2))))


;; Schema -----------------------------------------------------

;; TODO: Note this only handles hiccup-style malli schemas....
(defn io-schema?
  "Does the provide schema match the io schema?"
  [o]
  (m/validate spec/IOSchema o))


;; Catalogs & Definitions -------------------------------------

(defn flow-catalog? [c]
  (m/validate spec/FlowCatalog c))

(defn workflow-catalog? [c]
  (m/validate spec/WorkflowCatalog c))

(defn flows-definition? [c]
  (m/validate spec/FlowsDefinition c))

(defn system-catalog? [c]
  (m/validate spec/SystemCatalog c))

