(ns missinterpret.flows.spec
  (:require [manifold.stream :as s]
            [malli.core :as m]
            [malli.util :as mu]))

;; Utils ---------------------------------------------------------

(def Options
  [:*
   [:catn [:prop keyword?] [:value any?]]])


;; IO Schema -----------------------------------------------------

;; TODO: Note this only handles hiccup-style malli schemas....
(def IOSchema
  [:map
    [:schema/in     {:optional true} [:sequential some?]]
    [:schema/out    {:optional true} vector?]
    [:schema/throw? {:optional true} boolean?]
    [:schema/skip?  {:optional true} boolean?]])


;; Core Entities -------------------------------------------------

;; TODO This shouldn't have to be a keyword. Convert these to some? so that
;;      strings are supported....
;;
(def FlowBase
  [:map
   [:flow/id   keyword?]
   [:flow/uuid uuid?]
   [:flow/fn   {:gen/return identity} fn?]])

(def FlowOpts
  [:map
   [:flow/default-fn {:optional true :gen/return identity} fn?]
   [:flow/leave-fn   {:optional true :gen/return identity} fn?]])

(def Flow
  (mu/union
    (mu/required-keys FlowBase)
    (mu/optional-keys FlowOpts)))

(defn explain-flow
  [f]
  (m/explain Flow f))

(def WorkflowBase
  [:map
   [:workflow/id          keyword?]
   [:workflow/uuid        uuid?]
   [:workflow/definition  sequential?]])

(def WorkflowOpts
  [:map
   [:workflow/args           map?]
   [:workflow/order          keyword?]
   [:workflow/expand         boolean?]
   [:workflow/repeat-n       integer?]
   [:workflow/take-range     [:or integer? [:tuple integer? integer?]]]
   [:workflow/return-vector  boolean?]
   [:workflow/defer-load     boolean?]
   [:workflow/lazy-load      boolean?]
   [:workflow/new            boolean?]
   [:workflow/force          boolean?]
   [:workflow/return-error   boolean?]
   [:workflow/put-timeout    integer?]
   [:workflow/take-timeout   integer?]
   [:workflow/source         [:fn {:gen/return (s/stream)} s/stream?]]
   [:workflow/sink           [:fn {:gen/return (s/stream)} s/stream?]]
   [:workflow/invalid        [:sequential keyword?]]
   [:workflow/log-fn         [:and {:gen/return identity} fn?]]
   [:workflow/schema
    [:map
     [:allow-partial        {:optional true} boolean?]
     [:enforce-runtime-keys {:optional true} boolean?]
     [:minimum-rank         {:optional true} keyword?]
     [:required             {:optional true} boolean?]]]])


(def Workflow
  (mu/union
    (mu/required-keys
      WorkflowBase)
    (mu/optional-keys
      WorkflowOpts)))

(defn explain-workflow
  [wf]
  (m/explain Workflow wf))

(def Workflow-Factory-Success
  [:map
   [:workflow/source [:fn {:gen/return (s/stream)} s/stream?]]
   [:workflow/sink   [:fn {:gen/return (s/stream)} s/stream?]]])

(def Workflow-Factory-Failed
  [:map
   [:workflow/invalid [:sequential any?]]])

;; Schema Integrity

(def SchemaIntegrity
  [:map
   [:id        some?]
   [:rank      keyword?]
   [:graph     vector?]
   [:trace
     [:map
      [:no-schema set?]
      [:runtime   set?]
      [:passed    map?]
      [:conflict  map?]]]
   [:closed    boolean?]
   [:schema/in [vector?]]
   [:schema/out vector?]])


;; Flows Entities -------------------------------------------------

(def FlowDefinitions [:set Flow])
(def WorkflowDefinitions [:set Workflow])

(def FlowsDefinition
  [:map
   [:flows.flow.catalog/definitions     FlowDefinitions]
   [:flows.workflow.catalog/definitions WorkflowDefinitions]
   [:flows.workflow.catalog/args        {:optional true} map?]
   [:flows.pedestal/routes              {:optional true} [:set {:gen/return #{identity}} fn?]]])

(def FlowCatalog
  [:map-of :keyword Flow])
(def WorkflowCatalog
  [:map-of :keyword Workflow])

(def SystemCatalog
  [:map
   [:flow.catalog/loaded     FlowCatalog]
   [:workflow.catalog/loaded WorkflowCatalog]])


