(ns missinterpret.flows.system.catalog.workflow
  (:require [missinterpret.anomalies.anomaly :refer [throw+]]
            [missinterpret.flows.system.state :refer [state-atom] :as s]
            [missinterpret.flows.predicates :as pred.flows]
            [missinterpret.flows.utils :refer [opts]]))

;; Editing Fns ----------------------------------------------------------
;;

(defn catalog []
  (get @state-atom :workflow.catalog/loaded))

(defn ids []
  (->> (catalog) keys (into #{})))

(defn lookup [id & {:keys [throw-missing] :as op}]
  (let [wf (get-in @state-atom [:workflow.catalog/loaded id])]
    (if (and (true? throw-missing) (nil? wf))
      (throw+
        {:from     ::lookup
         :category :anomaly.category/invalid
         :message  {:readable (str id " not in catalog")
                    :reasons  [:invalid/id]
                    :data     {:id id :opts op}}})
      wf)))


(defn lookup-in [id path & {:keys [throw-missing] :as op}]
  (let [wf (get-in @state-atom (concat [:workflow.catalog/loaded id] path))]
    (if (and (true? throw-missing) (nil? wf))
      (throw+
        {:from     ::lookup-in
         :category :anomaly.category/invalid
         :message  {:readable (str id " not in catalog")
                    :reasons  [:invalid/id]
                    :data     {:id id :opts op}}})
      wf)))


(defn add!
  "Upserts a workflow to the catalog. If insert is true and the
   flow already exists an anomaly is thrown."
  [wf & {:keys [insert] :as op}]
  (cond
    (not (pred.flows/workflow? wf))
    (throw+
      {:from     ::add!
       :category :anomaly.category/invalid
       :message  {:readable (str wf " is not a workflow")
                  :reasons  [:invalid/argument]
                  :data     {:workflow wf :opts op}}})

    (and (some? (-> wf :workflow/id lookup))
         (-> (opts op) :insert true?))
    (throw+
      {:from     ::add!
       :category :anomaly.category/conflict
       :message  {:readable "Duplicate workflow encountered that conflicts with catalog"
                  :reasons  [:conflict/duplicate]
                  :data     {:workflow wf
                             :workflow.catalog/value (-> wf :workflow/id lookup)
                             :opts op}}})
    :else
    (let [id (:workflow/id wf)]
      (swap! state-atom assoc-in [:workflow.catalog/loaded id] wf))))

(defn remove!
  "Removes a workflow from the catalog"
  [id]
  (s/dissoc-in! [:workflow.catalog/loaded] id))
