(ns missinterpret.flows.core
  (:require [clojure.pprint :refer [pprint]]
            [manifold.stream :as s]
            [missinterpret.anomalies.anomaly :refer [throw+]]
            [missinterpret.flows.constructors :as cstr.flows]
            [missinterpret.flows.predicates :refer [flow? workflow? workflow-loaded? flow-catalog?]]
            [missinterpret.flows.utils :refer [try-put-all!] :as utils.flows]))

;; Catalog ----------------------------------------------------------------

(defn lookup-flow
  "Looks up a flow from the catalog by its :flow/id"
  [catalog id & {:keys [throw-missing] :or {throw-missing false}}]
  (if (flow-catalog? catalog)
    (let [flow (get catalog id)]
      (cond
        (and (nil? flow) (true? throw-missing))
        (throw+
          {:from     ::lookup-flow
           :category :anomaly.category/unavailable
           :message  {:readable (str id " not found")
                      :reasons  [:invalid.flow/id]
                      :data     {:flow-catalog catalog
                                 :id id}}})

        (and (some? flow) (not (flow? flow)))
        (throw+
          {:from     ::lookup-flow
           :category :anomaly.category/fault
           :message  {:readable (str id " not a valid flow")
                      :reasons  [:invalid/flow]
                      :data     {:flow-catalog catalog
                                 :flow flow
                                 :id id}}})
        :else flow))
    (throw+
      {:from     ::lookup-flow
       :category :anomaly.category/fault
       :message  {:readable "Not a valid flow catalog"
                  :reasons  [:invalid.flow/catalog]
                  :data     {:flow-catalog catalog
                             :id id}}})))


;; Flow  ---------------------------------------------------------

(defn fn-flow
  "Returns a flow map for a fn which wraps its execution to conform to the
   stream model."
  [fn-arg & {:flow/keys [expand id] :or {expand false id nil} :as opts}]
  (let [flow-id (keyword (if (some? id) id (str fn-arg)))]
    (if (true? expand)
      (cstr.flows/expand-flow flow-id fn-arg opts)
      (cstr.flows/fn-flow flow-id fn-arg opts))))

(defn flowify
  "Converts the argument into an appropriate flow.

   - flow or workflow: pass-through
   - fn: returns a fn-flow (optionally uses result pipelining via expand-flow)
   - else: lookup flow in catalog"
  [flow-catalog arg & {:flow/keys [expand] :or {expand false} :as opts}]
  (cond
    (flow? arg)     arg
    (workflow? arg) arg
    (fn? arg)       (fn-flow arg opts)
    :else           (lookup-flow flow-catalog arg)))


;; Workflow ------------------------------------------------------------

(defn validate-topology
  "Checks that the entries of the workflow definition to be processed by `workflow-factory`
   is a seq, isn't empty, has a recognized type, any dependent workflow is loaded,
   and exists in the flow catalog when defined by id."
  [flow-catalog definition]
  (let [validation
        (when (seq definition)
          (map
            (fn [x]
              (if (or (flow? x) (fn? x)
                      (and (workflow? x) (workflow-loaded? x))
                      (and (keyword? x) (lookup-flow flow-catalog x)))
                true
                x))
            definition))]
    (cond
      (nil? validation)         {:valid false :invalid [:empty-definition]}
      (every? true? validation) {:valid true}
      :else
      {:valid false
       :invalid (filterv #(not (boolean? %)) validation)})))


(defn linear-order
  [flow-catalog definition & {:keys [expand omit-leave] :or {expand false omit-leave false} :as opts}]
  (->> definition
       (map #(flowify flow-catalog % opts))
       (reduce
         (fn [coll f]
           (cond-> coll
             (contains? f :flow/default-fn)     (conj (cstr.flows/default-flow
                                                        (:flow/id f) (:flow/default-fn f) opts))
             true                               (conj f)
             (and (not (true? omit-leave))
                  (contains? f :flow/leave-fn)) (conj (fn-flow (:flow/leave-fn f) opts))))
         [])
       (filterv some?)))


;; http://pedestal.io/pedestal/0.7/guides/what-is-an-interceptor.html#_transition_from_enter_to_leave
;;
;; Calls leave in reverse order passing the context from the last enter fn
(defn interceptor-order [flow-catalog definition & {:keys [expand] :or {expand false} :as opts}]
  ;; 2 passes - first one through omitting leave-fn's then second one
  ;;  adds them at the end
  ;;
  ;;   - Note, not sure how to pass the opts correctly
  ;;   - I don't think this is quite right....
  (let [pass1 (linear-order flow-catalog definition (assoc opts :omit-leave true))]
    (reduce
      (fn [coll flow]
        (let [leave-fn (:flow/leave-fn flow)]
          (if (fn? leave-fn)
            (conj coll (fn-flow leave-fn opts))
            coll)))
      pass1
      (reverse pass1))))



(defn connect-node
  "Connects the sink of one flow as the source of the next
   updating the workflow.

   Note: This operates on the inverse of the convention
         for streams. i.e. source "
  [workflow node args]
  (try
    (let [source-for-fn (:workflow/source workflow)
          log-fn (:workflow/log-fn workflow)]
      (if (workflow? node)
        (let [sink (:workflow/sink node)
              source (:workflow/source node)]
          (utils.flows/safe-log log-fn {:from ::connect-node :message {:node node :workflow true}})
          (s/connect source-for-fn sink {:description (:workflow/id node)})
          (assoc workflow :workflow/source source))
        ;; The flow-fn takes args and a sink
        ;; returning its internal source
        (let [flow-fn (:flow/fn node)
              fn-src (flow-fn args source-for-fn)]
          (utils.flows/safe-log log-fn {:from ::connect-node :message {:node node :flow true}})
          (assoc workflow :workflow/source fn-src))))
    (catch Exception _ #:workflow{:invalid [node]})))


;; Runtime -----------------------------------------------------------------------

(defn invoke
  "Invokes the workflow operating on its streams to execute as a synchronous
   operation returning a vector of the results.

   Notes

   The operation is in two phases populating the sink with the passed argument
   using the workflow options and the take phase which populates the output
   vector.

   Unless specified by `return-n`, the operation will return all pending values from
   the workflow source.

   Options
    - expand       When true `run` will `put-all!` when the runtime argument is a seq
    - repeat-n     n copies of the passed argument is added before the take phase.
    - put-timeout  Will wait X milliseconds before throwing an exception during put operations.
    - take-timeout Will wait X milliseconds before throwing an exception during take operations."
  [{:workflow/keys [expand
                    repeat-n
                    put-timeout
                    take-timeout
                    sink source
                    log-fn]
    :or {expand false repeat-n nil} :as workflow}
   runtime-arg]
  (utils.flows/safe-log log-fn {:from ::invoke :message {:workflow workflow :runtime-arg runtime-arg}})
  ;; Put ---------------------------------------------------
  ;;
  (doseq [n-ignored (if (integer? repeat-n)
                      (range repeat-n)
                      [0])]

    (if (integer? put-timeout)

      ;; Timeout ---------------
      (if (true? expand)
        (let [_ (utils.flows/safe-log log-fn {:from ::invoke :message {:expand true :put-timeout put-timeout}})
              put-response (try-put-all! sink runtime-arg put-timeout ::timeout)]
          (utils.flows/safe-log log-fn {:from ::invoke :message {:expand true :put-response put-response}})
          (when (= put-response ::timeout)
            (throw+ {:from     ::invoke
                     :category :anomaly.category/busy
                     :message  {:readable "Put timed out"
                                :reasons  [:fault.run/busy]
                                :data     {:arg1 workflow :arg2 runtime-arg}}})))

        (let [put-response @(s/try-put! sink runtime-arg put-timeout ::timeout)]
          (utils.flows/safe-log log-fn {:from ::invoke :message {:expand false :put-response put-response}})
          (when (= put-response ::timeout)
            (throw+ {:from     ::invoke
                     :category :anomaly.category/busy
                     :message  {:readable "Put timed out"
                                :reasons  [:fault.run/busy]
                                :data     {:arg1 workflow :arg2 runtime-arg}}}))))

      ;; Immediate -----------
      (if (true? expand)
        (let [_ (utils.flows/safe-log log-fn {:from ::invoke :message {:expand true}})
              d (s/put-all! sink runtime-arg)]
          (utils.flows/safe-log log-fn {:from ::invoke :message {:expand true :put-all-response @d}})
          (when (false? @d)
            (throw+ {:from     ::invoke
                     :category :anomaly.category/busy
                     :message  {:readable "Put failed"
                                :reasons  [:fault.run/busy]
                                :data     {:arg1 workflow :arg2 runtime-arg}}})))
        (let [d (s/put! sink runtime-arg)]
          (utils.flows/safe-log log-fn {:from ::invoke :message {:put-all-response @d}})
          (when (false? @d)
            (throw+ {:from     ::invoke
                     :category :anomaly.category/busy
                     :message  {:readable "Put failed"
                                :reasons  [:fault.run/busy]
                                :data     {:arg1 workflow :arg2 runtime-arg}}}))))))

  ;; Take ----------------------------------------------
  ;;
  (let [buf-size (fn [] (-> (s/description source) :buffer-size))]
    (loop [results []]
      (cond
        (= 0 (buf-size)) results

        (integer? take-timeout)
        (let [_ (utils.flows/safe-log log-fn {:from ::invoke :message {:take-timeout take-timeout}})
              value @(s/try-take! source ::drained take-timeout ::timeout)]
          (utils.flows/safe-log log-fn {:from ::invoke :message {:take-timeout take-timeout :take-value value}})
          (if (or (= value ::timeout) (= value ::drained))
            (throw+ {:from     ::invoke
                     :category :anomaly.category/interrupted
                     :message  {:readable "Take timed out"
                                :reasons  [:fault/interrupted]
                                :data     {:arg1 workflow :arg2 runtime-arg}}})
            (recur (conj results value))))

        :else
        (let [value @(s/take! source ::drained)]
          (utils.flows/safe-log log-fn {:from ::invoke :message {:take-value value}})
          (if (= value ::drained)
            (throw+ {:from     ::invoke
                     :category :anomaly.category/interrupted
                     :message  {:readable "Source is drained"
                                :reasons  [:fault/interrupted]
                                :data     {:arg1 workflow :arg2 runtime-arg}}})
            (recur (conj results value))))))))



