(ns missinterpret.flows.xflow
  (:require [manifold.stream :as s]
            [missinterpret.flows.globals :as globals]
            [missinterpret.anomalies.anomaly :refer [anomaly anomaly?]]
            [missinterpret.flows.macros :refer [defn*]]
            [missinterpret.flows.schema :as schema]))

(defn* wrap-runtime-errors
  "Enforces that any function run in a workflow pipeline
   does not throw exceptions and that errors are propagated
   through the pipeline."
  {:schema/skip? true}
  [rt-fn element]
  (rt-fn element))

(defn default
  "Returns a callback fn compatible with xflow which merges the results of the `default-fn` into
   the argument."
  [default-fn]
  (let [schema (schema/io default-fn)]
    (cond-> (fn [sink element]
              (let [result (wrap-runtime-errors default-fn element)]
                (cond
                  (anomaly? result) (s/put! sink result)
                  (map? result)     (->> (merge result element) (s/put! sink))
                  :else
                  (->> (anomaly
                         ::default
                         :anomaly.category/fault
                         {:readable "Default fn did not return a map"
                          :data {:element element :default-fn default-fn}})
                       (s/put! sink)))))
            (map? schema) (with-meta schema))))


(defn ->fn [fn-arg]
  "Returns a callback fn compatible with xflow which adds the result of `fn-arg` to the sink via `put!`.
   If an exception is thrown it is wrapped as an anomaly and is returned. `core/invoke` will re-throw."
  (let [schema (schema/io fn-arg)]
    (cond-> (fn [sink element]
              (let [result (wrap-runtime-errors fn-arg element)]
                (s/put! sink result)))
            (map? schema) (with-meta schema))))


(defn fn-expand [fn-arg]
  "Returns a callback fn compatible with xflow which adds the result of `fn-arg` to the sink.
   Supports a single result or a sequence of values which are expanded by being added to
   the sink via `put-all!`"
  (let [schema (schema/io fn-arg)]
    (cond-> (fn [sink element]
              (let [result (wrap-runtime-errors fn-arg element)]
                (if (sequential? result)
                  (s/put-all! sink result)
                  (s/put! sink result))))
            (map? schema) (with-meta schema))))


(defn xflow
  "Returns a flow using `connect-via` with the passed xform functions
   which must have arity 2: [sink element] and return a function
   with the same arity.

    Optionally will add the default-fn."
  ([id xform]
   (let [schema (schema/io xform)
         flow-fn (cond-> (fn [_ sink]
                           (let [source (s/stream globals/buffer-size)]
                             (s/connect-via
                               sink
                               (partial xform source)
                               source
                               {:description id})
                             source))
                         (map? schema) (with-meta schema))
         flow #:flow{:id id
                     :uuid (random-uuid)
                     :fn flow-fn}]
     (if (map? schema) (with-meta flow schema) flow))))

