(ns missinterpret.flows.schema
  (:require [clojure.pprint :refer [pprint]]
            [clojure.main :refer (demunge)]
            [clojure.set :as set]
            [malli.generator :as mg]
            [malli.core :as m]
            [missinterpret.anomalies.anomaly :refer [throw+]]
            [missinterpret.flows.predicates :as pred.flows]
            [missinterpret.flows.spec :as spec]
            [missinterpret.flows.utils :as utils.flows]))

;; I/O Schema ------------------------------------------------------

(defn io
  "Returns the input/output schema of an object when
   added as metadata that matches the missinterpret.flows.spec/Defn*
   schema. (i.e. {:schema/in, :schema/out})

   Impl Note:

   When defn evaluates it generates a class with name as per (munge fn-symbol).
   The demunge function will reverse this operation.

   From the class name of a Fn object we can get the symbol back, and from the symbol,
   we look up the var which has the metadata attached to it."
  [o]
  (let [o-meta (meta o)]
    (if (some? o-meta)
      (utils.flows/extract o-meta spec/IOSchema)
      (try
        (let [m (-> o
                    class
                    print-str
                    demunge
                    symbol
                    find-var
                    meta
                    (utils.flows/extract spec/IOSchema))]
          (when-not (empty? m) m))
        (catch Exception _ nil)))))

(defn genable?
  [x]
  (try
    (mg/generate x)
    (catch Exception _ false)))


(defn compatible?
  "Does a value which conforms to the source schema
   validate against the destination schema?

   NOTE: The source schema MUST be able to generate
         a value via mg/generate."
  [src dest]
  (when (and (some? src) (some? dest))
    (try
      (let [src-gen (mg/generate src)]
        (m/validate dest src-gen))
      (catch Exception _ false))))


(defn to-map-format
  "Converts the malli schema into a map-type format"
  [schema]
  (m/walk
    schema
    (fn [schema _ children _]
      (-> (m/properties schema)
          (assoc :malli/type (m/type schema))
          (cond-> (seq children) (assoc :malli/children children))))))


;; Metadata ------------------------------------------------------

(defn add-meta
  "Adds the map keys to the existing metadata of the
   object."
  [o m]
  (let [o-meta (meta o)]
    (cond
      (some? o-meta) (with-meta o (merge o-meta m))
      (some? m)      (with-meta o m)
      :else o)))

(defn with-io
  "Attaches the io schema to the object"
  [o m]
  (if-let [s (utils.flows/extract m spec/IOSchema)]
    (add-meta o s)
    o))


;; Schema Definition Integrity --------------------------------------

(defn empty-trace []
  {:no-schema #{}
   :runtime   #{}
   :passed    {}
   :conflict  {}})

(defn score
  "Converts a rank into an integer that preserves
   the ordering."
  [rank]
  (case rank
    :conflict    1
    :runtime     2
    :passed      3
    :strict      4
    :no-schema   5
    1))

(defn rank
  "Converts the integer score into the keyword
   integrity rank"
  [score]
  (case score
    1 :conflict
    2 :runtime
    3 :passed
    4 :strict
    5 :no-schema
    :conflict))

(defn map-schema?
  [schema]
  (when (some? schema)
    (let [sm (to-map-format schema)]
      (= :map (:malli/type sm)))))

(defn closed?
  [schema]
  (when (some? schema)
    (let [sm (to-map-format schema)]
      (and  (contains? sm :closed) (true? (:closed sm))))))

(defn extract-integrity
  "If the workflow's metadata contains the schema integrity
   it is returned. Otherwise nil."
  [wf]
  (let [i (-> wf meta (utils.flows/extract spec/SchemaIntegrity))]
    (when (seq i) i)))

(defn merge-trace [trace {:keys [no-schema runtime passed conflict]}]
  (cond-> trace
          (seq no-schema) (assoc :no-schema (set/union (:no-schema trace) no-schema))
          (seq runtime)   (assoc :runtime (set/union (:runtime trace) runtime))
          (seq passed)    (assoc :passed (merge (:passed trace) passed))
          (seq conflict)  (assoc :conflict (merge (:conflict trace) conflict))))

(defn merge-traces
  "Collects and merges the `:no-schema`, `runtime`, `passed` and `conflict`
   traces from the chain."
  [chain]
  (reduce
    (fn [coll link]
      (merge-trace coll (:trace link)))
    (empty-trace)
    chain))


(defn trace-key
  "Traces the origin on the keyword to a previous
   entity in the chain.

   It searches by working backward from the last
   entity in the chain applying the heuristic:

   - Entity does not have a schema - skipped
   - Entity has a closed schema before key is found
     -> {:conflict #{ID-of-entity}}
        Note this does not conform to the general trace schema
   - A schema of an entity has the keyword
      -> {:passed {KW ID}}
   - The end of the chain is reached without finding the keyword
     -> {:runtime #{kw}}"
  [chain kw]
  (loop [links (reverse chain)]
    (let [l (first links)
          in-s (:schema/in l)
          in-keys (utils.flows/either (utils.flows/keys in-s) #{})
          in-closed (closed? in-s)
          out-s (:schema/out l)
          out-keys (utils.flows/either (utils.flows/keys out-s) #{})
          out-closed (closed? out-s)]
      (cond
        (nil? l) {:runtime #{kw}}

        (or (and (not (contains? out-keys kw)) (true? out-closed))
            (and (not (contains? in-keys kw)) (true? in-closed)))
        {:conflict #{(:id l)}}

        (or (contains? out-keys kw) (contains? in-keys kw))
        {:passed {kw (:id l)}}

        :else
        (recur (rest links))))))


(defn rank-flow
  [chain {:flow/keys [id] :as flow}]
  (let [{:schema/keys [in out] :as schema} (io flow)
        closed (or (closed? in) (closed? out))
        prev (last chain)
        prev-out-schema (:schema/out prev)
        comp (compatible? prev-out-schema in)
        trace (cond
                (or
                  (= 0 (count chain))
                  (true? comp))                         (empty-trace)
                (or (nil? in) (nil? out))               {:no-schema #{id}}
                (and (not (true? comp)) (true? closed)) {:conflict  {id (:id prev)}}
                (and (not (true? comp))
                     (not (map-schema? in)))            {:conflict  {id (:id prev)}}

                :else
                (let [prev-out-keys (utils.flows/keys prev-out-schema)
                      in-keys (utils.flows/keys in)
                      missing-keys (set/difference in-keys prev-out-keys)]
                  (reduce
                    (fn [coll x]
                      (let [t (trace-key chain x)
                            final-trace (if (contains? t :conflict)
                                          (assoc t :conflict {id (-> t :conflict first)})
                                          t)]
                        (merge-trace coll final-trace)))
                    (empty-trace)
                    missing-keys)))
        rank (cond
               (or
                 (and
                   (= 0 (count chain))
                   (some? schema))
                 (true? comp))            :strict
               (seq (:passed trace))      :passed
               (seq (:runtime trace))     :runtime
               (or
                 (seq (:no-schema trace))
                 (= 0 (count chain)))     :no-schema
               (seq (:conflict trace))    :conflict

               :else
               (throw+
                 {:from     ::rank-flow
                  :category :anomaly.category/fault
                  :message  {:readable "Can't calculate rank from traces and context"
                             :reasons  [:schema.integrity/fault]
                             :data     {:chain chain
                                        :flow flow
                                        :traces trace}}}))]
    (cond-> {:id      id
             :rank    rank
             :graph   chain
             :trace   trace
             :closed? closed}
            (some? schema) (merge #:schema{:in in :out out}))))


(defn unspecified?
  "Are all links in the chain :no-schema?"
  [chain]
  (every? #(= :no-schema (:rank %)) chain))

(defn underspecified?
  "Does the chain contain one non no-schema
   integrity rank?"
  [integrity]
  (let [chain (:graph integrity)]
    (and (not (unspecified? chain))
         (some #(= :no-schema (:rank %)) chain))))


(defn lowest-rank
  "Returns the lowest ranking in the chain
   of flow integrity entities.

   For example: [:strict :conflict]
   would return :conflict

   NOTE: Skips :no-schema when evaluating"
  [chain]
  (loop [entities chain
         integrity 5]
    (let [e (first entities)
          r (:rank e)
          s (score r)]
      (if (unspecified? chain)
        :no-schema
        (cond
          (nil? e)              (rank integrity)
          (and
            (not= :no-schema r)
            (< s integrity))    (recur (rest entities) s)
          :else
          (recur (rest entities) integrity))))))


(defn integrity
  [wf]
  (let [chain (reduce
                (fn [chain link]
                  (let [{:schema/keys [in out]} (io link)]
                    (cond
                      (and (some? in) (not (genable? in)))
                      (throw+
                        {:from     ::integrity
                         :category :anomaly.category/invalid
                         :message  {:readable "Input schema not generatable"
                                    :reasons  [:invalid/link]
                                    :data     {:workflow wf
                                               :schema/in in
                                               :link link}}})

                      (and (some? out) (not (genable? out)))
                      (throw+
                        {:from     ::integrity
                         :category :anomaly.category/invalid
                         :message  {:readable "Output schema not generatable"
                                    :reasons  [:invalid/link]
                                    :data     {:workflow wf
                                               :schema/out out
                                               :link link}}})

                      (pred.flows/flow? link) (conj chain (rank-flow chain link))

                      (pred.flows/workflow? link)
                      (let [wf-integrity (extract-integrity link)]
                        (if (seq wf-integrity)
                          (conj chain wf-integrity)
                          (conj chain (integrity link))))

                      :else (throw+
                              {:from     ::integrity
                               :category :anomaly.category/invalid
                               :message  {:readable "Workflow is invalid, entity not a flow or workflow"
                                          :reasons  [:invalid/entity]
                                          :data     {:workflow wf
                                                     :chain chain
                                                     :link link}}}))))
                []
                (:workflow/definition wf))
        rank (lowest-rank chain)
        trace (merge-traces chain)
        in-schema (->  chain first :schema/in)
        out-schema (-> chain last :schema/out)
        closed (or (closed? in-schema) (closed? out-schema))
        schema #:schema{:in in-schema :out out-schema}]
    (-> {:id      (:workflow/id wf)
         :rank    rank
         :graph   chain
         :closed? closed}
        (merge schema)
        (assoc :trace trace))))

