(ns missinterpret.flows.workflow
  (:require [clojure.pprint :refer [pprint]]
            [manifold.stream :as s]
            [missinterpret.anomalies.anomaly :refer [throw+ anomaly anomaly? wrap-exception]]
            [missinterpret.flows.globals :as globals]
            [missinterpret.flows.core :as core]
            [missinterpret.flows.predicates :as p]
            [missinterpret.flows.utils :refer [merge-if extract exclude]]
            [missinterpret.flows.predicates :refer [workflow? workflow-loaded?]]
            [missinterpret.flows.constructors :as cstr]
            [missinterpret.flows.spec :as spec]))

(defn workflow-factory
  "Converts a workflow definition into an entity which is ready for
   stream operations by using the returned :workflow/source and :workflow/sink
   arguments.

   Topology

   Determines the order of flow execution of the definition.
   The topology is linear unless another ordering is imposed.

   Options
    - :id           Adds the id to the workflow
    - :order        Sets the topology of the execution order

     :linear       (default) The topology is in the order provided by the definition vector
     :interceptor  The call stack will match the interceptor-chain calling order
                    when :flow/before-fn and/or :flow/after-fn exist in any of the
                    definitions flows.

    - :expand       Passes all the results from a sequential value into the downstream
                    flow individually."
  ([workflow]
   (workflow-factory workflow {}))
  ([workflow flow-args]
   (workflow-factory {} workflow flow-args))
  ([flow-catalog
   {:workflow/keys [definition id expand order]
    :or {id nil expand false order :linear} :as wf}
   {:flows/keys [source sink] :as factory-args}]
  (let [validation (core/validate-topology flow-catalog definition)]
    (if (:valid validation)
      (let [linear (cond
                     (= order :linear)      (core/linear-order flow-catalog definition :expand expand)
                     (= order :interceptor) (core/interceptor-order flow-catalog definition :expand expand))
            {:keys [src snk]}
            (cond
              (and (s/stream? source) (s/stream? sink)) {:src source :snk sink}
              (s/stream? source)                        {:src source :snk source}
              (s/stream? sink)                          {:src sink :snk sink}
              :else
              (let [snk-src (s/stream globals/buffer-size)] {:src snk-src :snk snk-src}))
            root-wf (cstr/workflow ::root [] :workflow/sink src :workflow/source snk)]
        (loop [nodes linear
               wf root-wf]
          (let [n (first nodes)]
            (if (nil? n)
              wf
              (let [loaded (core/connect-node wf n factory-args)]
                (if (contains? loaded :invalid)
                  loaded
                  (recur (rest nodes) loaded)))))))
      #:workflow{:invalid (:invalid validation)}))))


(defn load
  "Loads and connects the flows in the workflow definition adding
   the manifold source/sink making it compatible with run unless
   either `defer-load` and `lazy-load` are true.

   Note: The `force` option will always load the workflow definition."
  ([workflow]
   (load {} workflow))
  ([flow-catalog {:workflow/keys [default-fn defer-load lazy-load args force] :as rt-args}]
   (let [dfn (if (fn? default-fn) default-fn #())]
     (cond
       (not (-> rt-args (extract spec/Workflow) workflow?))
       (throw+
         {:from     ::load
          :category :anomaly.category/invalid
          :message  {:readable "Workflow is invalid"
                     :reasons  [:invalid/workflow]
                     :data     {:flow-catalog flow-catalog
                                :runtime-args rt-args}}})

       (or (true? force) (not (or (true? defer-load) (true? lazy-load))))
       (let [flow-args (-> rt-args
                           (exclude spec/Workflow)
                           (merge-if args)
                           (merge-if (dfn)))
             wf-rt (-> (extract rt-args spec/Workflow)
                       (dissoc :workflow/source)
                       (dissoc :workflow/sink))
             loaded (-> (workflow-factory flow-catalog wf-rt flow-args)
                        (merge wf-rt))]
         (if (p/workflow-loaded? loaded)
           loaded
           (throw+
             {:from     ::load
              :category :anomaly.category/invalid
              :message  {:readable "Workflow is invalid"
                         :reasons  [:invalid/workflow]
                         :data     {:flow-catalog flow-catalog
                                    :runtime-args rt-args
                                    :flow-args flow-args
                                    :loaded loaded}}})))

       :else (extract rt-args spec/Workflow)))))


(defn run
  "Executes a loaded workflow returning the results.

   Options:

   - return-n         Returns exactly n results from the workflow or fails if unable to do so
                      when timeouts are specified, otherwise run will hang.

                      NOTES:
                       - If no put/take timeout is set and the workflow stream has a 1-1 relationship
                         between input and output run will hang until the stream has been provided
                         the required input arguments

   - throw-on-error   When true any exception is returned as an anomaly.
                      Note that this will over-ride the`:workflow/throw` attribute if defined.

   - return-n         Returns exactly n results from the workflow or fails if unable to do so.
                      Note this option keeps the workflow 'loaded' which means the passed workflow can be
                      used with `run` again. This will over-ride the `:workflow/return-n attribute if defined.

   - repeat           When true and `return-n` is specified, `run` will `put!` n times before calling `take`.

   - expand           When true `run` will `put-all!` when the runtime argument is a seq.

   - put-timeout      Sets the amount of time `run` will wait before considering the operation failed.

   - put-timeout      Sets the amount of time `run` will wait before considering the operation failed."
  ([workflow]
   (run workflow {}))
  ([{:workflow/keys [return-n expand return-error] :as workflow} runtime-arg]
  (let [seq-result? (integer? return-n)]
    (cond

      (not (workflow? workflow))
      (let [anom {:from     ::run
                  :category :anomaly.category/invalid
                  :message  {:readable "Workflow is invalid"
                             :reasons  [:invalid/workflow]
                             :data     {:workflow workflow}}}]
        (if (true? return-error)
          (anomaly anom)
          (throw+ anom)))

      (not (workflow-loaded? workflow))
      (let [anom {:from     ::run
                  :category :anomaly.category/invalid
                  :message  {:readable "Workflow is not loaded"
                             :reasons  [:invalid.workflow/unloaded]
                             :data     {:workflow workflow}}}]
        (if (true? return-error)
          (anomaly anom)
          (throw+ anom)))

      ;; Expand; check arg is sequential
      (and (true? expand) (not (sequential? runtime-arg)))
      (let [anom {:from     ::run
                  :category :anomaly.category/invalid
                  :message  {:readable "Expand can only be used with a sequence of args"
                             :reasons  [:invalid.run/expand]
                             :data     {:workflow workflow
                                        }}}]
        (if (true? return-error)
          (anomaly anom)
          (throw+ anom)))

      :else
      (try
        (let [results (core/invoke workflow runtime-arg)]
          (if (some anomaly? results)
            (let [anoms (filter #(anomaly? %) results)
                  anom (if (= 1 (count anoms))
                         (first anoms)
                         (anomaly
                           {:from     ::run
                            :category :anomaly.category/fault
                            :message  {:readable "Run of workflow contained anomalies in the results"
                                       :reasons  [:fault.run/anomalies]
                                       :data     {:workflow workflow
                                                  :results results}}}))]
              (if (true? return-error)
                anom
                (throw+ anom)))
            (if seq-result?
              (vec results)
              (first results))))

        (catch Exception e (if (true? return-error)
                             (wrap-exception e ::run)
                             (throw e))))))))







