(ns missinterpret.flows.xflow
  (:require [manifold.stream :as s]
            [missinterpret.flows.globals :as globals]
            [missinterpret.anomalies.anomaly :refer [anomaly anomaly? wrap-exception]]
            [missinterpret.flows.macros :refer [defn*]]))

(defn* runtime-pipeline-constraints
  "Enforces that any function run in a workflow pipeline
   does not throw exceptions and that errors are propagated
   through the pipeline."
  {:schema/skip? true}
  [rt-fn element]
  (rt-fn element))

(defn default
  "Returns a callback fn compatible with xflow which merges the results of the `default-fn` into
   the argument."
  [default-fn]
  (fn [sink element]
    (let [result (runtime-pipeline-constraints default-fn element)]
      (cond
        (anomaly? result) (s/put! sink result)

        (map? result)
        (->> (merge result element) (s/put! sink))

        :else
        (->> (anomaly
               ::default
               :anomaly.category/fault
               {:readable "Default fn did not return a map"
                :data {:element element
                       :default-fn default-fn}})
             (s/put! sink))))))


(defn ->fn [fn-arg]
  "Returns a callback fn compatible with xflow which adds the result of `fn-arg` to the sink via `put!`.
   If an exception is thrown it is wrapped as an anomaly and is returned. `core/invoke` will re-throw."
  (fn [sink element]
    (let [result (runtime-pipeline-constraints fn-arg element)]
      (s/put! sink result))))


(defn fn-expand [fn-arg]
  "Returns a callback fn compatible with xflow which adds the result of `fn-arg` to the sink.
   Supports a single result or a sequence of values which are expanded by being added to
   the sink via `put-all!`"
  (fn [sink element]
    (let [result (runtime-pipeline-constraints fn-arg element)]
      (if (sequential? result)
        (s/put-all! sink result)
        (s/put! sink result)))))


(defn xflow
  "Returns a flow using `connect-via` with the passed xform functions
   which must have arity 2: [sink element] and return a function
   with the same arity.

    Optionally will add the default-fn."
  ([id xform]
   #:flow{:id id
          :uuid (random-uuid)
          :fn (fn [_ sink]
                (let [source (s/stream globals/buffer-size)]
                  (s/connect-via
                    sink
                    (partial xform source)
                    source
                    {:description id})
                  source))})
  ([id xform default-fn]
   (cond-> (xflow id xform)
           (fn? default-fn) (assoc :flow/default-fn default-fn))))

