(ns missinterpret.flows.system.catalog.flow
  (:require [clojure.pprint :refer [pprint]]
            [missinterpret.flows.system.state :refer [state-atom] :as s]
            [missinterpret.flows.predicates :as pred.flows]
            [missinterpret.anomalies.anomaly :refer [throw+]]
            [missinterpret.flows.utils :refer [opts]]))

;; Editing Fns ----------------------------------------------------------
;;

(defn catalog []
  (get @state-atom :flow.catalog/loaded))

(defn ids []
  (->> (catalog) keys (into #{})))

(defn lookup [id & {:keys [throw-missing] :as op}]
  (let [f (get-in @state-atom [:flow.catalog/loaded id])]
    (if (and (true? throw-missing) (nil? f))
      (throw+
        {:from     ::lookup
         :category :anomaly.category/invalid
         :message  {:readable (str id  " not in catalog")
                    :reasons  [:invalid/id]
                    :data     {:id id :opts op}}})
      f)))


(defn lookup-in [id path & {:keys [throw-missing] :as op}]
  (let [f (get-in @state-atom (concat [:flow.catalog/loaded id] path))]
    (if (and (true? throw-missing) (nil? f))
      (throw+
        {:from     ::lookup-in
         :category :anomaly.category/invalid
         :message  {:readable (str id " not in catalog")
                    :reasons  [:invalid/id]
                    :data     {:id id :opts op}}})
      f)))


(defn add!
  "Upserts a flow to the catalog. If insert is true and the
   flow already exists an anomaly is thrown."
  [flow & {:keys [insert] :as op}]
  (cond
    (not (pred.flows/flow? flow))
    (throw+
      {:from     ::add
       :category :anomaly.category/invalid
       :message  {:readable (str flow " is not a flow")
                  :reasons  [:invalid/argument]
                  :data     {:arg1 flow :opts op}}})

    (and (some? (-> flow :flow/id lookup))
         (-> (opts op) :insert true?))
    (throw+
      {:from     ::add!
       :category :anomaly.category/conflict
       :message  {:readable "Duplicate flow encountered that conflicts with catalog"
                  :reasons  [:conflict/duplicate]
                  :data     {:flow flow
                             :flow.catalog/value (-> flow :flow/id lookup)
                             :opts op}}})
    :else
    (let [id (:flow/id flow)]
      (swap! state-atom assoc-in [:flow.catalog/loaded id] flow))))

(defn remove!
  "Removes a flow from the catalog"
  [id]
  (s/dissoc-in! [:flow.catalog/loaded] id))

