(ns missinterpret.flows.system.workflow
  (:require [clojure.pprint :refer [pprint]]
            [com.stuartsierra.dependency :as dep]
            [missinterpret.anomalies.anomaly :refer [throw+]]
            [missinterpret.flows.predicates :as p]
            [missinterpret.flows.spec :as spec]
            [missinterpret.flows.utils :refer [extract exclude either]]
            [missinterpret.flows.workflow :as wf]))

;; Workflow Catalog -----------------------------------------------------

(defn throw-wf-invalid [wf-catalog wf]
  (cond
    (not (p/workflow? wf))
    (throw+
      {:from     ::throw-wf-invalid
       :category :anomaly.category/invalid
       :message  {:readable "Workflow is invalid"
                  :reasons  [:invalid/workflow]
                  :data     {:workflow wf :wf-catalog wf-catalog}}})

    (contains? wf-catalog (:workflow/id wf))
    (throw+
      {:from     ::throw-wf-invalid
       :category :anomaly.category/conflict
       :message  {:readable "Workflow already exists in catalog"
                  :reasons  [:fault.start/duplicate-flow]
                  :data     {:workflow wf :wf-catalog wf-catalog}}})
    :else wf))


(defn wf-skeleton
  "Creates the initial catalog skeleton. Verifies
   each workflow for structural validity and duplicates"
  [definitions]
  (reduce
    (fn [coll wf]
      (->> wf
           (throw-wf-invalid coll)
           (assoc coll (:workflow/id wf))))
    {}
    definitions))


;; Workflow DAG ----------------------------------------------------

(defn workflow-refs
  "Returns a seq of the workflow reference ids in the
   workflow definition or nil if none are present"
  [wf-catalog wf]
  (let [refs (filter #(and (keyword? %) (contains? wf-catalog %)) (:workflow/definition wf))]
    (when (seq refs) refs)))


(defn workflow-graph
  "Returns the dependency graph of the workflow definition
   as a seq of ids sorted in topological order.

   Validates that if the workflows definition has references
   that the DAG of its nested dependencies is not cyclic,
   throwing an exception when a back reference is encountered."
  ([wf-catalog wf graph]
   (try
     (if-let [def-refs (workflow-refs wf-catalog wf)]
       (loop [G graph
              refs def-refs]
         (let [id (first refs)]
           (if (nil? id)
             G
             (recur
               (->> (dep/depend G (:workflow/id wf) id)
                    (workflow-graph wf-catalog (get wf-catalog id)))
               (rest refs)))))
       graph)
     (catch Exception e (throw+
                          {:from     ::workflow-graph
                           :category :anomaly.category/conflict
                           :message  {:readable (.getMessage e)
                                      :reasons  [:conflict.workflow/cyclic-dependency]
                                      :data     {:workflow wf
                                                 :wf-catalog wf-catalog}}}))))
  ([wf-catalog wf]
   (->> (dep/graph)
        (workflow-graph wf-catalog wf)
        dep/topo-sort
        vec)))


(defn load
  "Returns a loaded workflow by preparing its definition
   to unpack any nested workflow references which aew
   then loaded.

   This is done by recursively loading any embedded references
   and replacing them in the embedding workflow definition
   with the loaded entity.

   Options:
    - flowify  Substitutes the flows from the embedded workflow
               definition for its loaded equivalent.

   Note:

   The returned definition in the top-level workflow
   will have any single workflow reference replaced with a new
   loaded workflow instance."
  [flow-catalog wf-catalog workflow & {:keys [flowify]}]
  (if (seq (workflow-refs wf-catalog workflow))
    (do
      ;; check for cyclic dependencies
      (workflow-graph wf-catalog workflow)
      (let [definition (mapv
                         (fn [id]
                           (cond
                             (contains? flow-catalog id) id

                             (contains? wf-catalog id)
                             (load flow-catalog wf-catalog (get wf-catalog id))

                             :else
                             (throw+
                               {:from     ::load
                                :category :anomaly.category/invalid
                                :message  {:readable (str id " missing from flow and workflow catalogs")
                                           :reasons  [:invalid.workflow/definition]
                                           :data     {:workflow workflow
                                                      :flow-catalog flow-catalog
                                                      :wf-catalog wf-catalog}}})))
                         (:workflow/definition workflow))]
        (wf/load flow-catalog (assoc workflow :workflow/definition definition))))

    ;; load the workflow
    (wf/load flow-catalog workflow)))


(defn try-load
  "Returns a load workflow if it is not marked as
   lazy/defer."
  [flow-catalog wf-catalog {:workflow/keys [force] :as wf}]
  (let [lazy? (true? (:workflow/lazy-load wf))
        defer? (true? (:workflow/defer-load wf))
        load? (or (true? force)
                  (and (false? lazy?) (false? defer?)))
        loaded (if (true? load?)
                 (load flow-catalog wf-catalog wf)
                 wf)]
    (if (and (true? load?) (p/workflow-loaded? loaded))
      loaded
      (throw+
        {:from     ::try-load
         :category :anomaly.category/fault
         :message  {:readable "Workflow failed to load when not marked defer or lazy"
                    :reasons  [:fault.start/unloaded]
                    :data     {:workflow loaded
                               :flow-catalog flow-catalog
                               :wf-catalog wf-catalog}}}))))

