(ns missinterpret.storage.store.internal
  (:require [clojure.pprint :refer [pprint]]
            [manifold.stream :as s]
            [missinterpret.anomalies.anomaly :refer [anomaly anomaly? wrap-exception]]
            [missinterpret.storage.block.core :as core.block]
            [missinterpret.storage.protocols.provider :as prot.content]
            [missinterpret.storage.protocols.block-store :as prot.block-store]
            [missinterpret.storage.protocols.content-store :as prot.content-store]
            [missinterpret.storage.block.predicate :refer [block?]]
            [missinterpret.storage.provider.core :as core.prov]
            [missinterpret.storage.provider.predicate :as pred.prov]
            [missinterpret.storage.provider.predicate :refer [provider?]]
            [missinterpret.storage.block.core :as block]
            [missinterpret.storage.provider.core :as provider]
            [missinterpret.storage.store.predicate :refer [consistent?]]))

;; Support Fns ------------------------------------------------------------
;;

(defn apply-default
  "If there is a provided default-fn apply it to the element.

  A default-fn must be arity-1 and return the modified element map"
  [default-fn element]
  (if (and (fn? default-fn) (map? element) (not (anomaly? element)))
    (try
      (default-fn element)
      (catch Exception _ element))
    element))

(defn write-block! [store-id
                    {:storage/keys [block]
                     :store.add/keys [metadata metadata-action rm-sources]
                     :as element}]
  (let [block-store (get store-id :store.id/block)
        content-store (get store-id :store.id/content)]
    (cond
      (anomaly? element) element

      (not (block? block))
      (anomaly
        ::write-block!
        :anomaly.category/invalid
        {:readable "Invalid argument: block missing"
         :reasons  [:invalid/block]
         :data     {:arg1 block-store :arg2 element}})

      (prot.block-store/exists? block-store element)
      (anomaly
        ::write-block!
        :anomaly.category/conflict
        {:readable "Conflict: block already exists"
         :reasons  [:exists/block]
         :data     {:arg1 block-store :arg2 element}})

      :else
      (let [write-block {:storage/block
                         (cond-> block
                                 (some? metadata)   (block/apply-metadata metadata metadata-action)
                                 (some? rm-sources) (block/rm-sources rm-sources))}
            stored-block (prot.block-store/add! block-store write-block)
            stored-provider (prot.content-store/provider content-store stored-block)]
        (cond
          (anomaly? stored-block)               stored-block
          (pred.prov/provider? stored-provider) stored-provider
          :else stored-block)))))


(defn write-content! [content-store {:storage/keys [block provider]
                                     :store.add/keys [error-when-content-exists?]
                                     :as element}]
  (cond
    (anomaly? element) element

    (not (provider? provider))
    (anomaly
      ::write-content!
      :anomaly.category/invalid
      {:readable "Invalid argument: content provider missing or invalid"
       :reasons  [:invalid/provider]
       :data     {:arg1 content-store :arg2 element}})

    (and
      (true? error-when-content-exists?)
      (prot.content-store/exists? content-store element))
    (anomaly
      ::write-content!
      :anomaly.category/conflict
      {:readable "Conflict: content exists"
       :reasons  [:exists/content]
       :data     {:arg1 content-store :arg2 element}})

    :else
    (let [source (if (prot.content-store/exists? content-store element)
                   (prot.content-store/source content-store element)
                   (prot.content-store/add! content-store element))
          stored-block (if (block? block)
                         (block/add-sources block #{source})
                         (let [metadata {}
                               address (provider/address provider)]
                           (block/direct-block metadata address source)))

          provider (prot.content-store/provider content-store {:storage/block stored-block})]
      (cond
        (anomaly? source)       source
        (anomaly? stored-block) stored-block
        (anomaly? provider)     provider
        :else                   (merge element provider)))))


(defn remove-block! [block-store {:storage/keys [block] :as element}]
  (cond
    (anomaly? element) element

    (not (block? block))
    (anomaly
      ::remove-block!
      :anomaly.category/invalid
      {:readable "Invalid argument: not a block"
       :reasons  [:invalid/block]
       :data     {:arg1 block-store :arg2 element}})

    (not (prot.block-store/exists? block-store element))
    (anomaly
      ::remove-block!
      :anomaly.category/conflict
      {:readable "Conflict: block missing"
       :reasons  [:missing/block]
       :data     {:arg1 block-store :arg2 element}})

    :else (prot.block-store/remove! block-store element)))


(defn remove-content! [content-store {:storage/keys [block provider] :as element}]
  (cond
    (anomaly? element) element

    (not (provider? provider))
    (anomaly
      ::remove-content!
      :anomaly.category/invalid
      {:readable "Invalid argument: content provider missing or invalid"
       :reasons  [:invalid/provider]
       :data     {:arg1 content-store :arg2 element}})

    (not (prot.content-store/exists? content-store element))
    (anomaly
      ::remove-content!
      :anomaly.category/conflict
      {:readable "Conflict: content does not exist"
       :reasons  [:missing/content]
       :data     {:arg1 content-store :arg2 element}})

    :else
    (let [source (prot.content-store/remove! content-store element)
          storage-block (if (some? block)
                          (block/rm-sources block #{source})
                          (let [provider-block (prot.content/block provider)]
                            (block/rm-sources provider-block #{source})))]
      (if (anomaly? source)
        source
        (assoc element :storage/block storage-block)))))


(defn change-availability! [content-store {:storage/keys [block provider] :as element}]
  (cond
    (anomaly? element) element

    (not (provider? provider))
    (anomaly
      ::change-availability!
      :anomaly.category/invalid
      {:readable "Invalid argument: content provider missing or invalid"
       :reasons  [:missing/provider]
       :data     {:content-store content-store :argument element}})

    (not (prot.content-store/exists? content-store element))
    (anomaly
      ::change-availability!
      :anomaly.category/conflict
      {:readable "Conflict: content missing"
       :reasons  [:missing/content]
       :data     {:arg1 content-store :arg2 element}})

    :else
    (let [source (prot.content-store/availability! content-store element)
          orig (prot.content/block provider)]
      (if (anomaly? source)
        source
        (assoc element :storage/block (if (block? block)
                                        (block/add-sources block #{source})
                                        (block/add-sources orig #{source})))))))


;; Invoke Fns -------------------------------------------------------------
;;

(defn query-fn
  "Returns a fn that is ready to process an element and return results.

   Passes the store-id to the implementing block store which
   generates a function that uses the content store."
  [store-id]
  (let [{:store.id/keys [block]} store-id
        query-args (dissoc store-id :store.id/block)]
    (prot.block-store/query block query-args)))


(defn add-fn
  "Returns a fn that is ready to process an element and return results."
  [store-id]
  (fn [{:store.add/keys [default-fn]
        :storage/keys [provider block] :as element}]

    (try
      (let [content-store (:store.id/content store-id)]
        (cond->> (apply-default default-fn element)
          (some? provider) (write-content! content-store)
          true             (write-block! store-id)))
      (catch Exception ex (wrap-exception ex ::add-fn)))))


(defn remove-fn
  "Returns a fn that is ready to process an element and return results."
  [store-id]
  ;;
  (fn [{:storage/keys [provider block]
        :store.remove/keys [default-fn]
        :as element}]
    (try
      (let [block-store (:store.id/block store-id)
            content-store (:store.id/content store-id)
            rm-element (apply-default default-fn element)]

        ;; Notes
        ;;  A. block only
        ;;     i. remove block
        ;;     ii. return removed block
        ;;  B. provider only
        ;;     i. remove content
        ;;     iV. return removed block - a clean block with just the removed source
        ;;  C. provider and block
        ;;     i. remove content
        ;;     ii. remove block
        ;;     iii. return removed block
        (cond
          (and (some? block) (nil? provider))
          (remove-block! block-store rm-element)

          (and (some? provider) (nil? block))
          (remove-content! content-store rm-element)

          (and (and (some? provider) (some? block))
               (not (consistent? rm-element)))
          (anomaly
            ::remove!
            :anomaly.category/conflict
            {:readable "Block and Content Provider not consistent"
             :reasons  [:conflict/inconsistent]
             :data     {:element element}})

          (and (some? provider) (some? block))
          (let [removed (remove-content! content-store rm-element)]
            (if (anomaly? removed)
              removed
              (remove-block! block-store rm-element)))

          :else
          (anomaly
            ::remove!
            :anomaly.category/invalid
            {:readable "Missing block or provider"
             :reasons  [:and :missing/block :missing/provider]
             :data     {:element element}})))

      (catch Exception ex (wrap-exception ex ::remove-fn)))))


;; TODO: These all supply a default-fn now, upgrade
(defn availability-fn
  "Returns a fn that is ready to process an element and return query results."
  [store-id]
  (fn [{:source.availability/keys [default]
        :storage/keys [provider] :as element}]
    (try
      (let [block-store (:store.id/block store-id)
            content-store (:store.id/content store-id)
            change (apply-default default element)
            orig (prot.content/block provider)
            changed (change-availability! content-store change)]
        (remove-block! block-store orig)
        (write-block! store-id changed))
      (catch Exception ex (wrap-exception ex ::availability-fn)))))


