(ns missinterpret.storage.store.content.memory
  (:require [clojure.pprint :refer [pprint]]
            [missinterpret.anomalies.anomaly :refer [throw+ anomaly anomaly?]]
            [missinterpret.storage.block.core :as core.block]
            [missinterpret.storage.block.predicate :as pred.block]
            [missinterpret.storage.protocols.content-store :as prot.content-store]
            [missinterpret.storage.protocols.provider :as prot.content]
            [missinterpret.storage.provider.memory :as provider.memory]
            [missinterpret.storage.provider.core :as core.prov]
            [missinterpret.storage.provider.predicate :as pred.prov]
            [missinterpret.storage.source.predicate :as pred.source]
            [missinterpret.storage.utils.byte-arrays :as utils.bytes]
            [missinterpret.storage.store.predicate :as pred.store]))

;; Definitions -------------------------------------------------------------
;;

(def version "1.0.0")
(def kind    :content-store.kind/memory)

(def content-store-info #:content-store.id{:version version
                                           :kind    kind})


;; Support Fns ------------------------------------------------------------
;;

(defn provider-source
  "Returns the source for this store from the content provider if it exists,
   returning nil otherwise."
  [{:content-store.id/keys [name] :as content-store-id}
   {:storage/keys [provider block] :as element}]
  (cond
    (pred.block/block? element) (-> (core.block/sources-by-name block name) first)
    (pred.prov/provider? provider) (-> (core.prov/sources-by-name provider name) first)))

(defn exists-in-store? [content-store-id element]
  (let [source (provider-source content-store-id element)]
    (try
      (pred.source/source? source)
      (catch Exception _ false))))

(defn get-format
  "Optional format for the sources in this store. Format is :no-encoding
   if not provided in the content store id."
  [content-store-id]
  (get-in content-store-id [:content-store.id/arguments :format] :no-encoding))

(defn add-content!
  "Returns a new source with the provider contents in memory.

   If format is provided in the content store id data is encoded."
  [content-store-id {:storage/keys [provider] :as element}]
  (when-not (pred.prov/provider? provider)
    (throw+
      {:from ::add-content!
       :reason :anomaly.category/invalid
       :message {:readable "Invalid provider"
                 :reasons  [:invalid/provider]
                 :data     {:arg1 content-store-id :arg2 element}}}))
  (let [name (:content-store.id/name content-store-id)
        {:storage/keys [input-stream data source]}  (prot.content/content provider element)
        {:source/keys [content-type ext]} source
        format (get-format content-store-id)
        source-data (cond
                      (some? data)
                      (utils.bytes/encode data format)

                      (some? input-stream)
                      (-> input-stream
                          utils.bytes/to-byte-array
                          (utils.bytes/encode format))

                      :else
                      (throw+
                        {:from ::add!
                         :category :anomaly.category/error
                         :message {:readable "Error - action on content provider failed to return input-stream or data"
                                   :reasons  [:storage.action.status/failed]
                                   :data     {:arg1 content-store-id :arg2 element}}}))]
    (provider.memory/new-source
      {:name         name
       :content-type content-type
       :ext          ext
       :format       format
       :data         source-data})))


;; Implementation ---------------------------------------------------------
;;

(defrecord MemoryContentStore [content-store-id]
  prot.content-store/ContentStore

  (id [_]  content-store-id)

  (info [_] content-store-info)

  ;; TODO: Documentation - this content store only supports provider/block due to its
  ;;       storage strategy
  (exists? [_ {:storage/keys [provider block] :as element}]
    (exists-in-store? content-store-id element))

  (provider [_ {:storage/keys [provider block] :as element}]
    (try
      (cond
        (and (pred.prov/provider? element)
             (exists-in-store? content-store-id element))
        provider

        (and (pred.block/block? element)
             (exists-in-store? content-store-id element))
        (provider.memory/new element))
      (catch Exception _ nil)))

  ;; TODO: Documentation - this content store only supports provider/block due to its
  ;;       storage strategy
  (source [_ {:storage/keys [provider block] :as element}]
    (provider-source content-store-id element))

  ;; TODO: Documentation - this returns a source
  (add! [_ {:storage/keys [provider] :as element}]
    (add-content! content-store-id element))

  ;; TODO: Documentation - this returns a source
  (remove! [this {:storage/keys [provider] :as element}]
    (when-not (pred.prov/provider? provider)
      (throw+
        {:from ::remove!
         :reason :anomaly.category/invalid
         :message {:readable "Invalid provider"
                   :reasons  [:invalid/provider]
                   :data     {:arg1 this :arg2 element}}}))
    (provider-source content-store-id element))

  (availability! [this arguments]
    (throw+
      {:from     ::availability!
       :category :anomaly.category/unavailable
       :message  {:readable "Not implemented"
                  :reasons  [:not-implemented]
                  :data     {:content-store this
                             :argument arguments}}})))


;; Factory ---------------------------------------------------------
;;

(defn compatible? [id]
  (pred.store/memory-content-store-id? id))

(defn id
  ([name]
   (id name nil))
  ([name format]
    (cond-> #:content-store.id{:name name}
            (some? format) (assoc-in [:content-store.id/arguments :format] format))))


(defn new [id]
  (if (pred.store/memory-content-store-id? id)
    (map->MemoryContentStore {:content-store-id id})
    (throw+
      {:from     ::new-content-store
       :category :anomaly.category/invalid
       :message  {:readable "Invalid ID for store content"
                  :reasons  [:invalid/content-store.id]
                  :data     {:arg1 id}}})))

