(ns missinterpret.storage.source.core
  (:require [clojure.pprint :refer [pprint]]
            [missinterpret.anomalies.anomaly :refer [throw+]]
            [missinterpret.storage.source.predicate :refer [source?]]
            [missinterpret.storage.utils.core :refer [upsert-ns]]))

(defn new [base]
  (let [source (upsert-ns base "source")]
    (if (source? source)
      (cond-> source
              (contains? source :source/uri) (assoc :source/uri
                                                    (-> source :source/uri str)))
      (throw+
        {:from     ::new-source
         :category :anomaly.category/invalid
         :message {:readable "Invalid arguments"
                   :reasons  [:invalid/args]
                   :data     {:arguments base}}}))))

;; Priority for source arguments:
;;
;;  Passed to content
;;   1. :storage/source
;;   2. :source/name
;;
;;  Passed as new argument
;;   3. passed :source/name
;;

(defn content-source
  "Used by content provider to select the source to
   load from.

   The function returns a source that matches the passed argument
   following the heuristic:

   0. throws an exception if the sources are empty
   1. :storage/source
   2. :source/name
   3. source-name
   4. No source found returns first source"
  [sources source-name {:storage/keys [source]
                        :source/keys  [name]
                        :as args}]
  (when (empty? sources)
    (throw+
      {:from     ::content-source
       :category :anomaly.category/invalid
       :message  {:readable "Source are empty"
                  :reasons  [:invalid/source]
                  :data     {:arg1 sources :arg2 source-name :arg3 args}}}))
  (let [name-fn (fn [n] (filter #(= n (:source/name %)) source))]
    (cond
      (and (some? source) (contains? sources source)) source

      (and (some? name) (seq (name-fn name))) (-> name name-fn first)

      (and (some? source-name) (seq (name-fn source-name))) (-> source-name name-fn first)

      :else (first sources))))

(defn name [source]
  (:source/name source))

(defn uri [source]
  (:source/uri source))