(ns missinterpret.storage.utils.byte-arrays
  (:require [clojure.java.io :as io]
            [clojure.pprint :refer [pprint]]
            [clj-cbor.core :as cbor]
            [clj-bencode.core :as enc]
            [clj-commons.byte-streams :as bts]
            [byte-transforms :as btr]
            [missinterpret.anomalies.anomaly :refer [throw+]])
  (:import java.util.Arrays
           (java.nio ByteBuffer)))

;; TODO: These encodings don't seem to preserve edn cleanly.
;;       TRANSIT does tho?
;; https://github.com/cognitect/transit-clj

;; Defs -----------------------------------------------------------------------
;;

(defonce encoding-formats #{:storage.byte-arrays.format/bytes
                            :storage.byte-arrays.format/cbor
                            :storage.byte-arrays.format/base64
                            :storage.byte-arrays.format/base64-bytes
                            :storage.byte-arrays.format/bzip2})

(defn supported? [format]
  (contains? encoding-formats format))



;; Encoding/Decoding ----------------------------------------------------------
;;

(defn decode
  "Attempts to decode the memory content in the storage block. If unable to be
   decoded this is pass-through."
  [data format]
  ;; If the data is read from a file then the encoded data is a
  ;; vector of int values. Pre-process the incoming data to catch
  ;; this edge case before attempting the format handling.
  (let [encoded (if (and (not (bytes? data)) (vector? data) (every? int? data))
                  (byte-array data)
                  data)]
    (cond
      ;; Encoded as a B] -> Whatever it decodes to
      ;; Note: that byte encoding raw edn does not preserve keywording.
      (= format :storage.byte-arrays.format/bytes)
      (enc/decode encoded)

      ;; Encoded as cbor format -> Whatever it decodes to
      ;; https://github.com/greglook/clj-cbor
      ;;
      (= format :storage.byte-arrays.format/cbor)
      (cbor/decode encoded)

      ;; Byte array encoded in base64 -> B] (byte array not decoded)
      (= format :storage.byte-arrays.format/base64)
      (btr/decode encoded :base64)

      ;; Encoded as B] packed in base64 -> decodes the base64 and the byte array
      (= format :storage.byte-arrays.format/base64-bytes)
      (-> encoded (btr/decode :base64) enc/decode)

      ;; Encoded as B] then Bzip2'ed  -> whatever data was packed into archive
      (= format :storage.byte-arrays.format/bzip2)
      (-> encoded (btr/decompress :bzip2) enc/decode)

      :else encoded)))


(defn encode
  "Attempts to encode the data given a specific format. If the format is not supported,
  the fn is pass-through."
  [data format]
  (cond
    ;; to B]
    (= format :storage.byte-arrays.format/bytes)
    (enc/encode data)

    ;; Encoded as cbor format
    ;; https://github.com/greglook/clj-cbor
    ;;
    (= format :storage.byte-arrays.format/cbor)
    (cbor/encode data)

    ;; Byte array encoded in base64 -> B] (byte array not decoded)
    (= format :storage.byte-arrays.format/base64)
    (btr/encode data :base64)

    ;; Encoded as B] packed in base64 -> decodes the base64 and the byte array
    (= format :storage.byte-arrays.format/base64-bytes)
    (-> data enc/encode (btr/encode :base64))

    ;; Encoded as B] then Bzip2'ed  -> whatever data was packed into archive
    (= format :storage.byte-arrays.format/bzip2)
    (-> data enc/encode (btr/compress :bzip2))

    :else data))


;; Transformations -------------------------------------------------------------
;;

(defn range-bytes
  "Returns a byte array representation of the memory content that respects
   the range if range arguments are supplied, pass through of content if not
   as byte-stream lib will do best conversion"
  [bytes start end]
  (if (or (some? start) (some? end))
    (cond
      (and (some? start) (some? end) (< end start))
      (throw+
        {:from     ::range-bytes
         :category :anomaly.category/invalid
         :message  {:readable (str "Invalid range: [" start "," end "]")
                    :reasons  [:invalid/bytes.range]
                    :data     {:arg1 bytes :arg2 start :arg3 end}}})

      :else (let [from  (if (some? start) start 0)
                  to    (if (some? end)
                  end   (count bytes))]

        ;; TODO: This seems ambigious
        (Arrays/copyOfRange bytes from to)))
    bytes))


(defn to-byte-array
  "Converts the input into a byte array:
    1. convert via clj-commons.byte-streams lib when there is a conversion-path
    2. Converts input to string then converts."
  [input]
  (cond
    (bytes? input) input

    (-> (type input)
        (bts/conversion-path ByteBuffer)
        seq)
    (bts/to-byte-array input)

    :else
    (-> (str input) bts/to-byte-array)))

