(ns missinterpret.storage.block.memory
  (:require [clojure.pprint :refer [pprint]]
            [missinterpret.anomalies.anomaly :refer [throw+]]
            [missinterpret.storage.block.predicate :as pred.block]
            [missinterpret.storage.protocols.block :as prot.block]
            [missinterpret.storage.block.core :as core.block]
            [missinterpret.storage.address.core :as core.address]
            [missinterpret.storage.provider.memory :as provider.memory]
            [missinterpret.storage.utils.byte-arrays :as utils.bytes]))

;; Definitions -------------------------------------------------------------
;;
(def kind    :block.kind/memory)
(def version "1.0.0")
(def hash    :sha2-256)

(def block-info #:block.info{:version version
                             :kind    kind
                             :hash    hash})


;; Support Fns -------------------------------------------------------------
;;

(defn data->content
  "Converts the passed data into the content part of the block.

  Note: If the format of the data is not explicitly passed using the
        available encoding types in missinterpret.storage.utils.byte-arrays/encoding-formats
        then the data is assigned without conversion.

        If the data is not edn-compatible then use with non memory storage
        will fail on read."
  [{:keys [data format] :as args}]
  (let [source-args (-> args
                        (dissoc :metadata)
                        (assoc :data (utils.bytes/encode data format)))]
    {:content/address (core.address/data->address data)
     :content/sources #{(provider.memory/new-source source-args)}}))


;; MemoryBlock Impl ----------------------------------------------------
;;

(defrecord MemoryBlock [arguments block address-cache]
  prot.block/Block

  (id [this]
    (let [address (if (some? @address-cache)
                    @address-cache
                    (let [addr (core.block/compute-id-address this)]
                      (reset! address-cache addr)
                      addr))]
      #:block.id{:address address
                 :arguments arguments}))

  (info [_] block-info)
  (data [_] block)
  (address [this] (core.block/address this))
  (metadata [this] (core.block/metadata this))
  (sources [this] (core.block/sources this)))


;; Factory ----------------------------------------------------------------
;;

(defn new
  "Creates a new memory block using passed data encoding it into the
   specified format.

   Note: If the format is not one of missinterpret.storage.utils.byte-arrays/encoding-formats
         the data is not changed."
  [{:keys [metadata] :as args}]
  (when-not (pred.block/memory-args? args)
    (throw+
      {:from     ::new
       :category :anomaly.category/invalid
       :message  {:readable "Arguments invalid"
                  :reasons  [:invalid/arguments]
                  :data     {:arguments args
                             :metadata metadata}}}))
  (let [meta (if (some? metadata) metadata {})]
    {:storage/block
     (map->MemoryBlock
       {:arguments args
        :block #:block{:metadata meta
                       :content  (data->content args)}
        :address-cache (atom nil)})}))

