(ns missinterpret.storage.store.core
  (:require [clojure.pprint :refer [pprint]]
            [missinterpret.anomalies.anomaly :refer [anomaly]]
            [missinterpret.storage.source.predicate :as pred.source]
            [missinterpret.storage.utils.core :as core.utils]
            [missinterpret.storage.protocols.store :as prot.store]
            [missinterpret.storage.protocols.block-store :as prot.block-store]
            [missinterpret.storage.protocols.content-store :as prot.content-store]
            [missinterpret.storage.store.predicate-runtime :as pred.store-rt]))

;; Store ----------------------------------------------------------
;;

(defn store-name
  ([store]
   (or (and
         (pred.store-rt/store? store)
         (-> (prot.store/id store) :store.id/name))
       (:store/name store)))
  ([block-store-id content-store-id]
   (let [b-name (-> block-store-id :block-store.id/uuid str)
         c-name (-> content-store-id :content-store.id/uuid str)]
     (keyword b-name c-name))))

(defn block-store-id
  [store]
  (try
    (-> (prot.store/id store)
        :store.id/block
        prot.block-store/id)
    (catch Exception _ nil)))

(defn content-store-id
  [store]
  (try
    (-> (prot.store/id store)
        :store.id/content
        prot.content-store/id)
    (catch Exception _ nil)))


;; Options --------------------------------------------------------
;;

(defn query-options
  "Includes the arguments that can modify a query."
  ([opts]
   (query-options {} opts))
  ([element {:keys [default-fn query-fn ordering-fn] :as opts}]
  (if (and (map? element)
           (or (fn? default-fn) (fn? query-fn) (fn? ordering-fn)))
    (->> (core.utils/upsert-ns opts "store.query")
         (merge element))
    (anomaly
      ::apply-query-options
      :anomaly.category/invalid
      {:readable "Invalid arguments"
       :reasons  [:invalid/options]
       :data     {:arg1 element :arg2 opts}}))))


(defn add-options
  "Includes the arguments that can modify a block during the add operation.

  When metadata is supplied without a metadata-action the action defaults
  to :metadata/merge-first."
  ([opts]
   (add-options {} opts))
  ([element {:keys [default-fn metadata metadata-action rm-sources] :as opts}]
  (if (and (map? element)
           (or (fn? default-fn) (map? metadata) (keyword? metadata-action)
               (and (set? rm-sources) (pred.source/are-sources? rm-sources))))
    (->> (core.utils/upsert-ns opts "store.add")
         (merge element))
    (anomaly
      ::apply-add-options
      :anomaly.category/invalid
      {:readable "Invalid arguments"
       :reasons  [:invalid/options]
       :data     {:arg1 element :arg2 opts}}))))


(defn remove-options
  ([opts]
   (remove-options {} opts))
  ([element {:keys [default-fn allow-content-only
                    omit-rm-block-predicate keep-orig-block-predicate]
             :as opts}]
  (if (and (map? element)
           (or (fn? default-fn) (true? allow-content-only)
               (fn? omit-rm-block-predicate) (fn? keep-orig-block-predicate)))
    (->> (core.utils/upsert-ns opts "store.remove")
         (merge element))
    (anomaly
      ::apply-remove-options
      :anomaly.category/invalid
      {:readable "Invalid arguments"
       :reasons  [:invalid/options]
       :data     {:arg1 element :arg2 opts}}))))

