(ns missinterpret.storage.store.predicate
  (:require [clojure.pprint :refer [pprint]]
            [malli.core :as m]
            [missinterpret.storage.block.core :as core.block]
            [missinterpret.storage.provider.core :as core.prov]
            [missinterpret.storage.store.spec :as spec]
            [missinterpret.storage.block.predicate :as pred.block]
            [missinterpret.storage.provider.predicate :as pred.prov]
            [missinterpret.storage.protocols.store :as prot.store]
            [missinterpret.storage.protocols.block-store :as prot.block-store]
            [missinterpret.storage.protocols.content-store :as prot.content-store]
            [missinterpret.storage.utils.file :as utils.file]))

;; IDs -------------------------------------------------

(defn block-store-id? [args]
  (m/validate spec/BlockStoreId args))

(defn memory-block-store-id? [args]
  (and (block-store-id? args)
       (m/validate spec/MemoryBlockStoreId args)))

(defn file-block-store-id? [args]
  (and (block-store-id? args)
       (let [directory (get-in args [:block-store.id/arguments :directory])]
         (utils.file/dir-exists? directory))
       (m/validate spec/FileBlockStoreId args)))

(defn content-store-id? [args]
  (m/validate spec/ContentStoreId args))

(defn memory-content-store-id? [args]
  (and (content-store-id? args)
       (m/validate spec/MemoryContentStoreId args)))

(defn file-content-store-id? [args]
  (and (content-store-id? args)
       (let [directory (get-in args [:content-store.id/arguments :directory])]
         (utils.file/dir-exists? directory))
       (m/validate spec/FileContentStoreId args)))

(defn consistent?
  "Does the provider's content address id match the block
   content address id?

   If either provider or block are not valid entities - returns true"
  [{:storage/keys [provider block]}]
  (try
    (cond
      (not (pred.prov/provider? provider)) true
      (not (pred.block/block? block)) true
      :else
      (let [addr-id-1 (-> (core.prov/address provider) :address/id)
            addr-id-2 (-> (core.block/address block) :address/id)]
        (= addr-id-1 addr-id-2)))
    (catch Exception _ false)))

;; Equivalence ---------------------------------------------

(defn block-store-id-equiv?
  "Do these two store ids share the same arguments?"
  [bs1 bs2]
  (let [args1 (get bs1 :block-store.id/arguments)
        args2 (get bs2 :block-store.id/arguments)]
    (= args1 args2)))

(defn content-store-id-equiv?
  "Do these two store ids share the same arguments?"
  [bs1 bs2]
  (let [args1 (get bs1 :content-store.id/arguments)
        args2 (get bs2 :content-store.id/arguments)]
    (= args1 args2)))

(defn block-store-equiv? [bs1 bs2]
  (try
    (let [id1 (prot.block-store/id bs1)
          id2 (prot.block-store/id bs2)]
      (block-store-id-equiv? id1 id2))))

(defn content-store-equiv? [cs1 cs2]
  (try
    (let [id1 (prot.content-store/id cs1)
          id2 (prot.content-store/id cs2)]
      (content-store-id-equiv? id1 id2))))

(defn store-id-equiv?
  [s1 s2]
  (let [bid1 (:store.id/block s1)
        cid1 (:store.id/content s1)
        bid2 (:store.id/block s2)
        cid2 (:store.id/content s2)]
    (and (block-store-equiv? bid1 bid2)
         (content-store-equiv? cid1 cid2))))

(defn store-equiv?
  [s1 s2]
  (try
    (let [sid1 (prot.store/id s1)
          sid2 (prot.store/id s2)]
      (store-id-equiv? sid1 sid2))
    (catch Exception _ false)))

(defn block-store? [block-store]
  (try
    (let [id (prot.block-store/id block-store)]
      (block-store-id? id))
    (catch Exception _ false)))

(defn content-store? [content-store]
  (try
    (let [id (prot.content-store/id content-store)]
      (content-store-id? id))
    (catch Exception _ false)))


;; Arguments -----------------------------------------------

(defn store-arguments? [args]
  (m/validate spec/StoreArgs args))

(defn flow-store-arguments? [args]
  (m/validate spec/FlowStoreArgs args))

(defn sync-store-arguments? [args]
  (m/validate spec/SyncStoreArgs args))


;; Fn Options ----------------------------------------------

(defn flow-store-opts?
  [opts]
  (m/validate spec/FlowStoreOpts opts))

(defn query-opts?
  [opts]
  (m/validate spec/QueryOpts opts))

(defn add-opts?
  [opts]
  (m/validate spec/AddOpts opts))

(defn remove-opts?
  [opts]
  (m/validate spec/RemoveOpts opts))

;; TODO: Availability Opts