(ns missinterpret.storage.utils.edn
  "Provides support for reading edn from provider content"
  (:require [clojure.edn :as edn]
            [missinterpret.anomalies.anomaly :refer [anomaly throw+ wrap-exception]]
            [missinterpret.storage.utils.core :as utils.core]
            [time-literals.read-write :as literals])
  (:import (java.io InputStreamReader PushbackReader)))

(defn read-string
  [{:storage/keys [data input-stream] :as content} & {:keys [throw-on-error]}]
  (cond
    (some? data) data

    (or
      (utils.core/input-stream? input-stream)
      (utils.core/input-stream? content))
    (with-open [r (if (some? input-stream)
                    (InputStreamReader. input-stream)
                    (InputStreamReader. content))]
      (try
        (edn/read
          {:readers literals/tags}
          (PushbackReader. r))
        (catch Exception e
          (let [wrapped (wrap-exception e ::read-string)]
            (if (true? throw-on-error) (throw wrapped) wrapped)))))

    :else
    (let [from ::read-string
          category :anomaly.category/invalid
          msg {:readable "Invalid content"
               :reasons  [:invalid/content]
               :data     {:arg1 content}}]
      (if (true? throw-on-error)
        (throw+
          {:from     from
           :category category
           :message  msg})
        (anomaly from category msg)))))

