(ns missinterpret.storage.store.block.file-system
  (:require [clojure.pprint :refer [pprint]]
            [clojure.java.io :as io]
            [missinterpret.anomalies.anomaly :refer [anomaly throw+ wrap-exception]]
            [missinterpret.storage.block.core :as core.block]
            [missinterpret.storage.block.file-system :as block.fs]
            [missinterpret.storage.protocols.block :as prot.block]
            [missinterpret.storage.protocols.block-store :as prot.block-store]
            [missinterpret.storage.protocols.content-store :as prot.content-store]
            [missinterpret.storage.store.block.core :as core.block-store]
            [missinterpret.storage.store.predicate :as pred.store]))

;; Definitions -------------------------------------------------------------
;;

(def version "1.0.0")
(def kind    :block-store.kind/file-system)

(def block-store-info #:block-store.info{:version version
                                         :kind    kind})


;; Support Fns ------------------------------------------------------------
;;

(defn block-directory [block-store-id]
  (get-in block-store-id [:block-store.id/arguments :directory]))

(defn block->output-file [block block-dir]
  (let [address-id (core.block/block-address-id block)]
    (io/file block-dir (str address-id ".edn"))))


(defn exists-in-store? [block-store-id {:storage/keys [block]}]
  (try
    (->> (block-directory block-store-id)
         (block->output-file block)
         .exists)
    (catch Exception _ false)))


(defn load-store-element [content-store file]
  (try
    (let [block (block.fs/load {:file file :use-path-address? true})
          provider (when (pred.store/content-store? content-store) (prot.content-store/provider content-store block))]
      (cond-> block
        (some? provider) (merge provider)))
    (catch Exception ex (wrap-exception ex ::load-store-element))))


(defn query-store [block-store-id {:store.id/keys [content]}]
  (fn [{:store.query/keys [query-fn ordering-fn default-fn] :as element}]
    (if-not (pred.store/query-opts? element)
      (anomaly
        ::query
        :anomaly.category/invalid
        {:readable "Invalid argument"
         :reasons  [:invalid/arguments]
         :data     {:arg1 element}})
      (try
        (let [block-dir (block-directory block-store-id)
              blocks (->> block-dir
                          io/file
                          .listFiles
                          block.fs/id-files
                          (map #(load-store-element content %)))]
          (cond->> blocks

                   (fn? default-fn)
                   (map #(default-fn %))

                   (core.block-store/valid-query-fn? query-fn)
                   (query-fn)

                   (fn? ordering-fn)
                   (sort ordering-fn)))
        (catch Exception e (wrap-exception e ::query))))))


(defn write-block! [block-store-id {:storage/keys [block] :as element}]
  (let [block-dir (block-directory block-store-id)
        output-file (block->output-file block block-dir)
        block-data (-> block prot.block/data pprint with-out-str)]
    (spit output-file block-data :append false)
    element))


(defn remove-block! [block-store-id {:storage/keys [block] :as element}]
  (let [block-dir (block-directory block-store-id)
        output-file (block->output-file block block-dir)]
    (io/delete-file output-file)
    element))


;; Implementation ---------------------------------------------------------
;;

(defrecord FileSystemBlockStore [block-store-id]
  prot.block-store/BlockStore

  (id [_] block-store-id)

  (info [_] block-store-info)

  (exists? [_ {:keys [:storage/block] :as element}]
    (exists-in-store? block-store-id element))

  (query [_ {:keys [:storage.store.provider/content] :as element}]
    (query-store block-store-id element))

  (add! [_ {:keys [:storage/block] :as element}]
    (write-block! block-store-id element))

  (remove! [_ {:keys [:storage/block] :as element}]
    (remove-block! block-store-id element)))


;; Factory ---------------------------------------------------------
;;

(defn compatible? [id]
  (pred.store/file-block-store-id? id))


(defn id [name dir]
  #:block-store.id{:name name
                   :arguments {:directory dir}})


(defn new [id]
  (if (compatible? id)
    (map->FileSystemBlockStore {:block-store-id id})
    (throw+
      {:from     ::new
       :category :anomaly.category/invalid
       :message  {:readable "Invalid ID for block store"
                  :reasons  [:invalid/block-store.id]
                  :data     {:arg1 id}}})))
