(ns missinterpret.storage.store.block.memory
  (:require [clojure.pprint :refer [pprint]]
            [missinterpret.anomalies.anomaly :refer [anomaly throw+ wrap-exception]]
            [missinterpret.storage.block.core :as core.block]
            [missinterpret.storage.protocols.block-store :as prot.block-store]
            [missinterpret.storage.protocols.content-store :as prot.content-store]
            [missinterpret.storage.store.block.core :as core.block-store]
            [missinterpret.storage.store.predicate :as pred.store]))

;; Definitions -------------------------------------------------------------
;;

(def version "1.0.0")
(def kind    :block-store.kind/memory)

(def block-store-info #:block-store.info{:kind    kind
                                         :version version})


;; Support Fns ------------------------------------------------------------
;;

(defn store-atom [block-store-id]
  (get-in block-store-id [:block-store.id/arguments :store-atm]))

(defn exists-in-store? [block-store-id {:storage/keys [block]}]
  (let [store-atm (store-atom block-store-id)
        address-id (core.block/block-address-id block)]
    (contains? @store-atm (keyword address-id))))

(defn load-store-element [content-store block]
  (let [block {:storage/block block}
        provider (when (pred.store/content-store? content-store) (prot.content-store/provider content-store block))]
    (cond-> block
      (some? provider) (merge provider))))


(defn query-store
  "Returns a function which implements querying functionality."
  [block-store-id {:store.id/keys [content]}]
  (fn [{:store.query/keys [query-fn ordering-fn default-fn] :as element}]
    (if-not (pred.store/query-opts? element)
      (anomaly
        ::query
        :anomaly.category/invalid
        {:readable "Invalid argument"
         :reasons  [:invalid/arguments]
         :data     {:arg1 element}})
      (try
        (let [store-atm (store-atom block-store-id)
              blocks (map (fn [[_ v]] (load-store-element content v)) @store-atm)]
          (cond->> blocks

                   (fn? default-fn)
                   (map #(default-fn %))

                   (core.block-store/valid-query-fn? query-fn)
                   (query-fn)

                   (fn? ordering-fn)
                   (sort ordering-fn)))
        (catch Exception e (wrap-exception e))))))


(defn add-block!
  "Adds the block to the store atom"
  [block-store-id {:storage/keys [block] :as element}]
  (let [store-atm (store-atom block-store-id)]
    (swap! store-atm assoc (keyword (core.block/block-address-id block)) block)
    element))


(defn remove-block!
  "Removes the block from the store atom"
  [block-store-id {:storage/keys [block] :as element}]
  (let [store-atm (store-atom block-store-id)]
    (swap! store-atm dissoc (keyword (core.block/block-address-id block)))
    element))


;; Implementation ---------------------------------------------------------
;;

(defrecord MemoryBlockStore [block-store-id]
  prot.block-store/BlockStore

  (id [_] block-store-id)

  (info [_] block-store-info)

  (exists? [_ element]
    (exists-in-store? block-store-id element))

  (query [_ element]
    (query-store block-store-id element))

  (add! [_ element]
    (add-block! block-store-id element))

  (remove! [_ element]
    (remove-block! block-store-id element)))


;; Factory ---------------------------------------------------------
;;

(defn compatible? [args]
  (pred.store/memory-block-store-id? args))


(defn id
  ([]
   (id :storage.block-store/memory))
  ([name]
   (id name (atom {})))
  ([name store-atm]
   #:block-store.id{:name name
                    :arguments {:store-atm store-atm}}))


(defn new [id]
  (if (compatible? id)
    (if (contains? id :block-store.id/arguments)
      (map->MemoryBlockStore {:block-store-id id})
      (let [atm-id (assoc id :block-store.id/arguments {:store-atm (atom {})})]
        (map->MemoryBlockStore {:block-store-id atm-id})))
    (throw+
      {:from     ::new
       :category :anomaly.category/invalid
       :message  {:readable "Invalid ID for memory block store"
                  :reasons  [:invalid/block-store.id]
                  :data     {:arg1 id}}})))

