(ns missinterpret.storage.utils.core
  (:require [clojure.pprint :refer [pprint]]
            [clojure.string :as s]
            [clojure.java.io :as io]
            [clj-commons.byte-streams :as bts]
            [clojurewerkz.urly.core :as url])
  (:import (java.io File)
           (java.net URI)
           (java.util ArrayList)
           (java.io ByteArrayOutputStream InputStream)))

;; URIs -----------------------------------------------------------------
;;

(defn uri-inst? [uri]
  (= java.net.URI (type uri)))

(defn uri-matches? [uri protocol]
  (= protocol (-> (url/url-like uri)  url/protocol-of)))

(defn to-uri [x]
  (if (uri-inst? x)
    x
    (URI. x)))

(defn uri? [uri]
  (try
    (to-uri uri)
    (catch Exception _ false)))


;; Maps -----------------------------------------------------------------
;;

(defn strip-nil
  "Strips any keys with nil values from a map"
  [m]
  (reduce
    (fn [coll k]
      (if-let [value (get m k)]
        (assoc coll k value)
        coll))
    {}
    (keys m)))


(defn strip-ns
  "Strips the namespace from all top-level
   keys in a map"
  [m]
  (->> (reduce
          (fn [coll k]
            (let [value (get m k)]
              (assoc coll (keyword (name k)) value)))
          {}
          (keys m))
       strip-nil))


(defn upsert-ns
  "Adds or replaces the namespace of all top-level
   keys in a map"
  [m namespace-name]
  (->> (reduce
          (fn [coll k]
            (let [value (get m k)]
              (assoc coll (keyword namespace-name (name k)) value)))
          {}
          (keys m))
       strip-nil))


(defn file-values->uri-string
  "Postwalk's a map to replace all file values with string uris"
  [m]
  (let [f (fn [v]
            (if (instance? File v)
              (str (.toURI v))
              v))]
    (clojure.walk/postwalk f m)))

;; Language Macros ----------------------------------------------------
;;

(defmacro when-let*
  ([bindings & body]
   (if (seq bindings)
     `(when-let [~(first bindings) ~(second bindings)]
        (when-let* ~(drop 2 bindings) ~@body))
     `(do ~@body))))


;; Transducers ---------------------------------------------------------
;;

(defn xf-sort
  "A sorting transducer. Mostly a syntactic improvement to allow composition of
  sorting with the standard transducers, but also provides a slight performance
  increase over transducing, sorting, and then continuing to transduce."
  ([]
   (xf-sort compare))
  ([cmp]
   (fn [rf]
     (let [temp-list (ArrayList.)]
       (fn
         ([]
          (rf))
         ([xs]
          (reduce rf xs (sort cmp (vec (.toArray temp-list)))))
         ([xs x]
          (.add temp-list x)
          xs))))))


;; Input Stream --------------------------------------------------------
;;

(defn input-stream?
  [in]
  (-> in type str (s/ends-with? "InputStream")))


(defn input-stream->byte-array [in]
  (with-open [out (ByteArrayOutputStream.)]
    (io/copy in out)
    (.toByteArray out)))

(defn to-input-stream
  "Converts data to an input stream when there is a conversion path.
   Returns the data unchanged otherwise."
  [data]
  (if (-> data type (bts/conversion-path InputStream) seq)
    (bts/to-input-stream data)
    data))

(defn slurp-input-stream
  [is]
  (with-open [r (io/reader is)] (slurp r)))

