(ns missinterpret.storage.block.s3
  (:require [clojure.pprint :refer [pprint]]
            [clojure.edn :as edn]
            [missinterpret.anomalies.anomaly :refer [throw+ throw-if-cognitect-anomaly]]
            [missinterpret.storage.block.core :as core.block]
            [missinterpret.storage.block.predicate :as pred.block]
            [missinterpret.storage.protocols.block :as prot.block]
            [missinterpret.storage.utils.s3 :as utils.s3]))

;; Definitions -------------------------------------------------------------
;;

(def kind    :block.kind/s3)
(def version "1.0.0")
(def hash :sha2-256)

(def block-info #:block.info{:version version
                             :kind    kind
                             :hash    hash})

;; ContentProvider Impl ----------------------------------------------------
;;

(defrecord S3Block [arguments block-cache]
  prot.block/Block

  (id [this]
    (let [{:keys [use-key-address? validate-key-address?]} arguments
          key-id (utils.s3/key-id arguments)]
      (if (true? use-key-address?)
        ;; TODO: It would be great to cache the object size and last-modified date
        ;;       when the object is loaded as part of data access....
        ;;   - Otherwise validating the object
        (let [address #:address{:id            key-id
                                :algorithm     hash
                                :size          (utils.s3/object-size arguments)
                                :calculated-at (utils.s3/object-last-modified arguments)}]
          (when (and (true? validate-key-address?)
                     (not= key-id (-> (core.block/compute-id-address this) :address/id)))
            (throw+
              {:from     ::id
               :category :anomaly.category/incorrect
               :message  {:readable "File address id does not match computed address id"
                          :reasons  [:incorrect/s3.address]
                          :data     {:arg1 arguments}}}))
          #:block.id{:address   address
                     :arguments arguments})
        (let [address (core.block/compute-id-address this)]
          (when (and (true? validate-key-address?)
                     (not= key-id (:address/id address)))
            (throw+
              {:from     ::id
               :category :anomaly.category/incorrect
               :message  {:readable "Key address id does not match computed address id"
                          :reasons  [:incorrect/s3.address]
                          :data     {:arg1 arguments}}}))
          #:block.id{:address   address
                     :arguments arguments}))))

  (info [_] block-info)

  (data [_]
    (if (some? @block-cache)
      @block-cache
      (let [sb (-> (utils.s3/get-object arguments)
                   throw-if-cognitect-anomaly
                   :Body
                   slurp
                   edn/read-string)]
        (reset! block-cache sb)
        sb)))

  (address [this] (core.block/address this))
  (metadata [this] (core.block/metadata this))
  (sources [this] (core.block/sources this)))


;; Factory ----------------------------------------------------------------
;;

(defn load [arguments]
  (when-not (pred.block/s3-args? arguments)
    (throw+
      {:from     ::new-block
       :category :anomaly.category/invalid
       :message  {:readable "Block arguments invalid"
                  :reasons  [:invalid/block.arguments]
                  :data     {:arg1 arguments}}}))
  {:storage/block
   (map->S3Block {:arguments arguments
                  :block-cache (atom nil)})})

