(ns missinterpret.storage.store.constructor
  (:require [clojure.pprint :refer [pprint]]
            [missinterpret.anomalies.anomaly :refer [throw+]]
            [missinterpret.storage.store.predicate :as pred.store]
            [missinterpret.storage.store.core :as core.store]
            [missinterpret.storage.store.block.memory :as block-store.memory]
            [missinterpret.storage.store.block.file-system :as block-store.fs]
            [missinterpret.storage.store.content.memory :as content-store.memory]
            [missinterpret.storage.store.content.file-system :as content-store.fs]))

(defn new-block-store [args]
  (cond
    (pred.store/memory-block-store-id? args)
    (block-store.memory/new args)

    (pred.store/file-block-store-id? args)
    (block-store.fs/new args)

    :else
    (throw+
      {:from     ::new
       :category :anomaly.category/unsupported
       :message  {:readable "Unsupported Block Store"
                  :reasons  [:unsupported/block-store]
                  :data     {:arg1 args}}})))


(defn new-content-store [args]
  (cond
    (pred.store/memory-content-store-id? args)
    (content-store.memory/new args)

    (pred.store/file-content-store-id? args)
    (content-store.fs/new args)

    :else
    (throw+
      {:from     ::new
       :category :anomaly.category/unsupported
       :message  {:readable "Unsupported Content Store"
                  :reasons  [:unsupported/content-store]
                  :data     {:arg1 args}}})))


(defn new
  "Returns a new store entity throwing an exception if the
   arguments are invalid."
  [{:store/keys [name kind] :as args}]
  (when-not (pred.store/store-arguments? args)
    (throw+
      {:from     ::new
       :category :anomaly.category/invalid
       :message  {:readable "Store arguments invalid"
                  :reasons  [:invalid/arguments]
                  :data     {:arg1 args}}}))

  (let [block-store (new-block-store (get-in args [:store/arguments :block]))
        content-store (new-content-store (get-in args [:store/arguments :content]))]
    (cond
      (nil? kind)
      (core.store/new-store name block-store content-store)

      (= kind :store.kind/flow)
      (core.store/new-flow-store name block-store content-store)

      :else
      (throw+
        {:from     ::new
         :category :anomaly.category/unsupported
         :message  {:readable "Unsupported Store"
                    :reasons  [:invalid/arguments]
                    :data     {:arg1 args}}}))))

