(ns specql.transform
  "Define custom transformation to be done for a field."
  (:refer-clojure :exclude [Keyword ->Keyword])
  (:require [clojure.spec.alpha :as s]))

(defprotocol Transform
  (from-sql [this sql-value]
    "Transform value from SQL result set. This is called before query returns a value.")
  (to-sql [this transformed-value]
    "Transform value back to SQL for update or insert.")
  (transform-spec [this input-spec]
    "Generate a spec for the transformed value. The input-spec is the spec that
  would be generated by default."))

(s/def ::transform #(satisfies? Transform %))
(s/def ::transform-definition (s/keys :req [::transform]))

(s/fdef transform
        :args ::transform
        :ret ::transform-definition)

(defn transform
  "Return a transform definition for the given Transform instance."
  [transform]
  {::transform transform})

(defrecord Keyword [ns]
  Transform
  (from-sql [_ sql-value]
    (keyword ns (str sql-value)))
  (to-sql [_ keyword-value]
    (when keyword-value
      (name keyword-value)))
  (transform-spec [_ input-spec]
    (if (set? input-spec)
      ;; This is an enum of values, make them all keywords
      (into #{}
            (map #(keyword ns (str %)))
            input-spec)

      ;; Don't know what this is (could be text or varchar), just return keyword
      keyword?)))

(defn to-keyword
  "Return a transformation that converts read strings to keyword and back again.
  Optionally takes a namespace for the generated keywords."
  ([] (to-keyword nil))
  ([ns] (->Keyword ns)))
