(ns io.github.theasp.simple-encryption
  (:require
   [clojure.string :as str]
   [cljsjs.forge :as forge]
   [cognitect.transit :as t]
   [taoensso.timbre :as timbre
    :refer-macros (tracef debugf infof warnf errorf)]))

(def profile-0
  {:cipher             :aes-256-cbc
   :digest-type        :sha-512-256
   :data-encoding      :utf8
   :encrypted-encoding :base64
   :key-encoding       :bytes
   :kdf                :pbkdf2
   :kdf-salt-encoding  :bytes
   :kdf-salt-size      128
   :kdf-iterations     10000
   })

(def default-profile profile-0)

(def cipher-params
  {:aes-256-cbc {:key-size 32 :iv-size 16 :cipher-name "AES-CBC"}
   :aes-192-cbc {:key-size 24 :iv-size 16 :cipher-name "AES-CBC"}
   :aes-128-cbc {:key-size 16 :iv-size 16 :cipher-name "AES-CBC"}
   :3des-cbc    {:key-size 24 :iv-size 8  :cipher-name "DES-CBC"}
   :des-cbc     {:key-size 8  :iv-size 8  :cipher-name "DES-CBC"}})

(def digests
  {:md5         {:create js/forge.md5           :size 16}
   :sha1        {:create js/forge.sha1          :size 20}
   :sha-256     {:create js/forge.sha256        :size 32}
   :sha-384     {:create js/forge.sha384        :size 48}
   :sha-512     {:create js/forge.sha512        :size 64}
   :sha-512-224 {:create js/forge.sha512.sha224 :size 28}
   :sha-512-256 {:create js/forge.sha512.sha256 :size 32}})


(defn bytes->hex
  "Convert `data` from bytes to hex."
  [data]
  (when data
    (js/forge.util.bytesToHex data)))

(defn hex->bytes
  "Convert `data` from hex to bytes."
  [data]
  (when data
    (js/forge.util.hexToBytes data)))

(defn bytes->base64
  "Convert `data` from bytes to base64"
  [data]
  (when data
    (js/forge.util.encode64 data)))

(defn base64->bytes
  "Convert `data` from base64 to bytes."
  [data]
  (when data
    (js/forge.util.decode64 data)))

(defn bytes->utf8
  "Convert `data` from bytes to utf8."
  [data]
  (when data
    (js/forge.util.encodeUtf8 data)))

(defn utf8->bytes
  "Convert `data` from utf8 to bytes."
  [data]
  (when data
    (js/forge.util.decodeUtf8 data)))

(defn bytes->encoding
  "Convert `data` from bytes to `encoding`, where `encoding` is one of
  `:hex`, `:base64`, `:utf8`, or `:bytes`."
  [data encoding]
  (condp = encoding
    :hex (bytes->hex data)
    :base64 (bytes->base64 data)
    :utf8 (bytes->utf8 data)
    :bytes data
    (throw (js/Error. (str "Unknown encoding type: " encoding)))))

(defn encoding->bytes
  "Convert `data` from `encoding` to bytes, where `encoding` is one of
  `:hex`, `:base64`, `:utf8`, or `:bytes`."
  [data encoding]
  (condp = encoding
    :hex (hex->bytes data)
    :base64 (base64->bytes data)
    :utf8 (utf8->bytes data)
    :bytes data
    (throw (js/Error. (str "Unknown encoding type: " encoding)))))

(defn buffer
  "Create a new forge buffer, optionally supplying `data` and a `type`
  which is either `:raw` or `:utf8`"
  [& [data type]]
  (if data
    (if type
      (js/forge.util.createBuffer data (name type))
      (js/forge.util.createBuffer data))
    (js/forge.util.createBuffer)))

(defn random-bytes
  "Generate `size` random bytes"
  [size]
  (js/forge.random.getBytesSync size))

(defn digest
  "Generate a digest of `data` using `digest`, where `digest` is one
  of `:md5`, `:sha1`, `:sha-256`, `:sha-384`, `:sha-512`,
  `:sha-512-224`, or `:sha-512-256`.  Optionally supply
  `ouput-encoding` (defaults to `:hex`) or `input-encoding` (defaults
  to `:utf`) with any encoding supported by `bytes->encoding`"
  [data digest-type & [{:keys [input-encoding output-encoding] :as poop}]]
  (let [input-encoding (or input-encoding :utf8)
        output-encoding (or output-encoding :hex)
        md-type (:create (get digests digest-type))]
    (assert (some? md-type)
            (str "Unknown digest type: " digest-type))
    (let [data (encoding->bytes data input-encoding)]
      (-> (.create md-type)
          (.update data)
          (.digest)
          (.bytes)
          (bytes->encoding output-encoding)))))

(defn- cipher-data
  "Apply a cipher using `cipher` and `data` from options, `key` as the
  encryption key, `iv` as the intial vector (random if nil), and
  `mode` as `:encrypt` or `:decrypt`."
  [data cipher mode key iv]

  (tracef "cipher-data: Applying cipher: Cipher=%s Mode=%s Key=[%s] %s IV=[%s] %s Data=[%s] %s"
          cipher
          mode
          (count key)
          (bytes->base64 key)
          (count iv)
          (bytes->base64 iv)
          (count data)
          (bytes->base64 data))

  ;;(debugf "%s" data)

  (let [{:keys [key-size iv-size cipher-name] :as cipher-opts} (get cipher-params cipher)]
    (assert (some? cipher-opts)
            (str "Unknown cipher: \"" cipher "\""))

    (assert (= key-size (count key))
            (str "The supplied key is not the correct length for " cipher ": Need " key-size " given " (count key)))

    (let [iv (or iv (random-bytes iv-size))
          forge-cipher
          (if (= mode :encrypt)
            (js/forge.cipher.createCipher cipher-name key)
            (js/forge.cipher.createDecipher cipher-name key))]
      (assert (= iv-size (count iv))
              (str "The supplied IV is not the correct length for \"" cipher "\": Need " iv-size " given " (count iv)))

      (.start forge-cipher #js {:iv iv})
      (.update forge-cipher (buffer data :raw))
      (.finish forge-cipher)

      (let [data (js/forge-cipher.output.bytes)]
        (if (= :encrypt mode)
          {:data (js/forge-cipher.output.bytes)
           :iv iv}
          (js/forge-cipher.output.bytes))))))

(defprotocol IEncryption
  "For deriving encryption engines"
  (encrypt-with* [this data options] "See `encrypt-with`")
  (decrypt-with* [this data options] "See `decrypt-with`"))

(defn encrypt-with
  "Encrypt `data` using `key`. Optionally, supply a a map, `options`,
   which can have any of the following:

   :data-encoding (default :utf8)
   The encoding of the data in it's unencrypted form, from the
   supported encoding types.  Defaults to :utf8.  See `encoding->bytes`
   for details.

   :encrypted-encoding (default :base64)
   The encoding of the data in it's encrypted form, from the supported
   encoding types.  See `bytes->encoding` for details.

   This function returns a map which contains all the information
   needed to decrypt the data, except for the encryption key.  For all
   key types the encrypted data will be in in `:data`, and the initial
   vector in `:iv`.  These entries will be encoded using the
   `:encrypted-encoding`.  Different types of keys will have more
   entries, for instance a `Pbkdf2` key will have `:kdf-salt`,
   `:kdf-iterations` and `:kdf`"
  [key data & [options]]
  (encrypt-with* key data options))

(defn decrypt-with
  "Decrypt `data` using `key` to produce the decrypted output. `data`
   must contain all the entries in the map that `encrypt-with`
   produced. Optionally, supply a map named `options` which can have
   any of the following:

   :data-encoding (default :utf8)
   The encoding of the data in it's unencrypted form, from the
   supported encoding types.  Defaults to :utf8.  See `encoding->bytes`
   for details.

   :encrypted-encoding (default :base64)
   The encoding of the data in it's encrypted form, from the supported
   encoding types.  See `bytes->encoding` for details.

   This function will return the decrypted data encoded using `:data-encoding`."
  [key data & [options]]
  (decrypt-with* key data options))

(defprotocol IKdf
  "For Key Deriviation Functions"
  (generate-key [this salt iterations key-size]
    "Generate a `StaticKey` using given `salt`, `iterations`, and
    `key-size`"))

(defrecord StaticKey [key cipher]
  Object
  (toString [_] (str "[StaticKey:" (bytes->base64 key) "]"))

  IEncryption
  (encrypt-with* [this data options]
    (let [{:keys [data-encoding encrypted-encoding] :as options}
          (merge default-profile options)]
      (-> data
          (encoding->bytes data-encoding)
          (cipher-data cipher :encrypt key nil)
          (update :data bytes->encoding encrypted-encoding)
          (update :iv bytes->encoding encrypted-encoding)
          (assoc :cipher cipher)
          (assoc :encoding encrypted-encoding))))

  (decrypt-with* [this data options]
    (let [{:keys [data-encoding encrypted-encoding] :as options}
          (merge default-profile options)

          encrypted-encoding (or (:encoding data) encrypted-encoding)
          cipher (or (:cipher data) cipher)

          iv (encoding->bytes (:iv data) encrypted-encoding)]
      (when iv
        (-> (:data data)
            (encoding->bytes encrypted-encoding)
            (cipher-data cipher :decrypt key iv)
            (bytes->encoding data-encoding))))))

(defn new-static-key [key cipher]
  (let [key-size (get-in cipher-params [cipher :key-size])]
    (assert (some? key-size)
            (str "Unknown cipher: " cipher))
    (assert (= key-size (count key))
            (str "Supplied key is not the correct size for cipher: " key " " cipher))
    (StaticKey. key cipher)))

(defn new-random-key [cipher]
  (let [key-size (get-in cipher-params [cipher :key-size])]
    (assert (some? key-size)
            (str "Unknown cipher: " cipher))
    (new-static-key (random-bytes key-size) cipher)))

(defrecord Pbkdf2 [password cipher]
  Object
  (toString [_] (str "[Pbkdf]"))

  IKdf
  (generate-key [this salt iterations key-size]
    (let [salt (or salt (random-bytes 128))
          iterations (or iterations 10000)]
      (when (and salt iterations key-size)
        (new-static-key (js/forge.pkcs5.pbkdf2 password salt iterations key-size) cipher))))

  IEncryption
  (encrypt-with* [this data options]
    (let [{:keys [encrypted-encoding kdf-salt-size kdf-iterations] :as options}
          (merge default-profile options)

          salt     (random-bytes kdf-salt-size)
          key-size (get-in cipher-params [cipher :key-size])
          key      (generate-key this salt kdf-iterations key-size)]
      (-> (encrypt-with* key data options)
          (assoc :kdf :pbkdf2
                 :kdf-salt (bytes->encoding salt encrypted-encoding)
                 :kdf-iterations kdf-iterations))))

  (decrypt-with* [this data options]
    (let [{:keys [encrypted-encoding cipher kdf-salt kdf-iterations] :as options}
          (merge default-profile options)

          cipher     (or (:cipher data) cipher)
          key-size   (get-in cipher-params [cipher :key-size])
          salt       (or (:kdf-salt data) kdf-salt)
          salt       (encoding->bytes salt encrypted-encoding)
          iterations (or (:kdf-iterations data) kdf-iterations)]
      (when (and salt iterations)
        (let [key (generate-key this salt iterations key-size)]
          (debugf "Pbkdf2.decrypt-with: Key=%s" key)
          (decrypt-with* key data options))))))

(defn new-pbkdf2 [password cipher]
  "Create a new PBKDF2 based key based on `password` to use with
  `cipher`.  The actual key will not be generated until `encrypt-with`
  or `decrypt-with` are called."
  (Pbkdf2. password cipher))


(defn encrypt-container-with
  "Produce a container to hold encrypted data which is accessable
  using one or many keys, where `access-keys` is a single key or a
  collection of keys and `data` is the data to be encrypted.
  Interally the container will have `data` encrypted with a randomly
  generated key, which is stored encrypted using each  of the
  `access-keys`.  In addition a hash is generated and encrypted.  You
  can provide `options` with the following keys:

  :data-encoding (default :utf8)
  The encoding of the data in it's unencrypted form, from the
  supported encoding types.  Defaults to :utf8.  See `encoding->bytes`
  for details.

  :cipher (default :aes-256-cbc)

  :digest-type (default :sha-512-256)"
  [access-keys data & [options]]

  (let [{:keys [data-encoding digest-type cipher] :as options}
        (merge default-profile options {:encrypted-encoding :base64})

        data
        (encoding->bytes data data-encoding)

        options
        (assoc options :data-encoding :bytes)

        inside-key
        (new-random-key cipher)

        access-keys
        (if (seq? access-keys) access-keys [access-keys])

        cur-digest
        (digest data digest-type {:input-encoding :bytes :output-encoding :bytes})]
    (t/write (t/writer :json)
             {:version 0
              :access-keys
              (map #(encrypt-with* % (:key inside-key) options) access-keys)

              :data
              (encrypt-with* inside-key data options)

              :digest
              (assoc (encrypt-with* inside-key cur-digest options) :type digest-type)})))

(defn decrypt-container-with
  "Consume a container which holds encrypted data using a single
  `unlock-key`, which is used to decrypt a random key internally,
  which is then used to decrypt the data and the digest.  The digest
  is used to ensure that the decrypted data is what was encrypted.
  The following `options` are supported:

  :data-encoding (default :utf8)"
  [unlock-key data & [options]]

  (let [{:keys [data-encoding] :as options}
        (merge default-profile options)

        options
        (assoc options :data-encoding :bytes)

        data (t/read (t/reader :json) data)

        cipher (get-in data [:data :cipher])
        key-size (get-in cipher-params [cipher :key-size])

        access-keys (:access-keys data)

        data-digest
        (-> (:digest data)
            (assoc :encoding :bytes)
            (update :data base64->bytes)
            (update :iv base64->bytes))

        digest-type
        (:type data-digest)

        data
        (-> (:data data)
            (assoc :encoding :bytes)
            (update :data base64->bytes)
            (update :iv base64->bytes))]

    (assert (some? key-size)
            (str "Unknown cipher: " cipher))

    (let [attempt-decrypt
          (fn [inside-key]
            (let [inside-key (decrypt-with unlock-key inside-key options)]
              (when (= key-size (count inside-key))
                (let [inside-key
                      (new-static-key inside-key cipher)

                      decrypted-data
                      (decrypt-with inside-key data options)

                      decrypted-digest
                      (decrypt-with inside-key data-digest options)

                      current-digest
                      (digest decrypted-data digest-type {:input-encoding :bytes :output-encoding :bytes})]
                  (when (= current-digest decrypted-digest)
                    decrypted-data)))))]
      (loop [access-keys access-keys]
        (when-let [access-key (first access-keys)]
          (if-let [decrypted (attempt-decrypt access-key)]
            decrypted
            (recur (rest access-keys))))))))


(defn encrypt-container-with-password [password data & [options]]
  (let [{:keys [cipher] :as options}
        (merge default-profile options)
        access-key (new-pbkdf2 password cipher)]
    (encrypt-container-with access-key data options)))

(defn decrypt-container-with-password [password data & [options]]
  (let [{:keys [cipher] :as options}
        (merge default-profile options)
        access-key (new-pbkdf2 password cipher)]
    (decrypt-container-with access-key data options)))
