;; Copyright © technosophist
;;
;; This Source Code Form is subject to the terms of the Mozilla Public
;; License, v. 2.0. If a copy of the MPL was not distributed with this
;; file, You can obtain one at http://mozilla.org/MPL/2.0/.
;;
;; This Source Code Form is "Incompatible With Secondary Licenses", as
;; defined by the Mozilla Public License, v. 2.0.
(ns systems.thoughtfull.decoder
  (:require
    [clojure.java.io :as io])
  (:import
    (clojure.lang LineNumberingPushbackReader)
    (java.io PushbackReader)))

(defn pushback-reader
  "Ensures in is a java.io.PushbackReader."
  ^PushbackReader [in]
  (if (instance? PushbackReader in)
    in
    (PushbackReader. (io/reader in))))

(defn line-numbering-pushback-reader
  "Ensures in is a clojure.lang.LineNumberingPushbackReader."
  ^LineNumberingPushbackReader [in]
  (if (instance? LineNumberingPushbackReader in)
    in
    (LineNumberingPushbackReader. (io/reader in))))

(defn make
  "Returns a constructor for a reducible on a data source that, opens the source, repeatedly decodes
  data from it, and closes it when it is done."
  ([open decode done?]
   (make open decode done? java.lang.AutoCloseable/.close))
  ([open decode done? close]
   (fn [source]
     (reify
       clojure.lang.IReduceInit
       (reduce
         [_ rf init]
         (let [source (open source)]
           (try
             (loop [acc init
                    v (decode source)]
               (if (done? v)
                 acc
                 (let [acc (rf acc v)]
                   (if (reduced? acc)
                     @acc
                     (recur acc (decode source))))))
             (finally
               (when close
                 (close source))))))))))

(defmacro defdecoder
  "Defines a new decoder with name and optional docstring.  See make."
  {:arglists '([name docstring? open decode done?] [name docstring? open decode done? close])}
  [name & args]
  (let [[docstring? & args] (if (string? (first args))
                              args
                              (cons nil args))
        [open decode done?] args
        close (if (= (count args) 4)
                (nth args 3)
                java.lang.AutoCloseable/.close)]
    `(def ~(vary-meta name
             merge
             {:arglists ''([source])}
             (when docstring? {:doc docstring?}))
       (make ~open ~decode ~done? ~close))))
