(ns mulify.http
  (:require [clojure.data.xml :as dx]
            [clojure.string :as str]
            [mulify.dataweave :as dw]
            [mulify.tls :as tls]
            [silvur.util :refer [uuid map->json]]
            [mulify.core :refer (defprocess camelize)]))



(defprocess listener-connection
  {:port "8081" :host "localhost" :protocol "HTTP"})

(defprocess listener-config {:name "listner_config"})

(defn https-listener-config* [& {:keys [name host port key-type key-path key-password]
                                 :or {name "listener_config"
                                      key-type "pkcs12"
                                      host "0.0.0.0"
                                      port "8082"}}]
  (listener-config
   {:name name}
   (listener-connection
    {:host host :port port :protocol "HTTPS"}
    (tls/context
     (tls/trust-store {:type key-type})
     (tls/key-store
      {:type key-type
       :path key-path
       :key-password key-password
       :password key-password}))))  
  )

(defn http-listener-config* [& {:keys [name host port] :or {name "listener_config" port "8081"}}]
  (->> (listener-connection {:host host :port port})
       (listener-config {:name name})))




(defprocess request-connection {:host "localhost" :port "80" :protocol "HTTP"})

(defprocess request-config {:name "request_config"})


(defn http-request-config* [& {:keys [name port host]
                               :or {name "http_request_config" host "localhost" port "80"}}]
  (request-config {:name name}
                  (request-connection {:protocol "HTTP" :host host :port port})))

(defn https-request-config* [& {:keys [name port host]
                                :or {name "https_request_config" host "localhost" port "443"}}]
  (request-config {:name name}
                  (request-connection {:protocol "HTTPS" :host host :port port})))

(defprocess response {:status-code "#[vars.httpStatus default 200]"})
(defprocess error-response {:status-code "#[vars.httpStatus default 500]"})
(defprocess body)
(defprocess listener {:path "/api/*" :config-ref "http_listner_config"})

(defprocess request {:path "/",
                     :config-ref "request_config"
                     :method "GET"})

(defprocess query-params :update-fn dx/cdata)

(defprocess headers)

(defprocess load-static-resource {:resource-base-path "${app.home}/public"})

(defn listener* [{:keys [config-ref path]
                  :or {config-ref "listener_config" path "/api/*"}}]
  (listener
   {:config-ref config-ref, :path path}
   (response
    {:status-code "#[attributes.statusCode default 200]"}
    (headers "#[attributes.headers default {}]"))
   (error-response
    {:status-code "#[vars.statusCode default 500]"}
    (body "#[payload]")
    (headers "#[vars.headers default {}]"))))
